% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/02b_Lab_color_clustering.R
\name{getLabHist}
\alias{getLabHist}
\title{Generate a 3D histogram based on CIE Lab color coordinates in an image}
\usage{
getLabHist(
  image,
  bins = 3,
  sample.size = 10000,
  ref.white,
  from = "sRGB",
  bin.avg = TRUE,
  alpha.channel = TRUE,
  as.vec = FALSE,
  plotting = TRUE,
  lower = c(0, 0.55, 0),
  upper = c(0.24, 1, 0.24),
  title = "path",
  a.bounds = c(-128, 127),
  b.bounds = c(-128, 127),
  ...
)
}
\arguments{
\item{image}{Path to a valid image (PNG or JPG) or a \code{\link{loadImage}}
object.}

\item{bins}{Number of bins for each channel OR a vector of length 3 with bins
for each channel. Bins = 3 will result in 3^3 = 27 bins; bins = c(2, 2, 3) will
result in 2 * 2 * 3 = 12 bins (2 L, 2 a, 3 b), etc.}

\item{sample.size}{Numeric. How many pixels should be randomly sampled from
the non-background part of the image and converted into CIE Lab
coordinates? If non-numeric, all pixels will be converted, but this can be
very slow (see details).}

\item{ref.white}{Reference white passed to \code{\link{convertColorSpace}}.
Unlike \code{convertColor}, no default is provided. See details for
explanation of different reference whites.}

\item{from}{Original color space of image, probably either "sRGB" or "Apple
RGB", depending on your computer.}

\item{bin.avg}{Logical. Should the returned color clusters be the average of
the pixels in that bin (bin.avg=\code{TRUE}) or the center of the bin
(\code{FALSE})? If a bin is empty, the center of the bin is returned as the
cluster color regardless.}

\item{alpha.channel}{Logical. If available, should alpha channel transparency be
used to mask background? See \code{\link{removeBackground}} for more
details.}

\item{as.vec}{Logical. Should the bin sizes just be returned as a vector?
Much faster if only using \code{\link{chisqDistance}} for comparison metric.}

\item{plotting}{Logical. Should a histogram of the bin colors and sizes be
plotted?}

\item{lower, upper}{RGB or HSV triplets specifying the lower and upper bounds
for background pixels. Default upper and lower bounds are set to values
that work well for a bright green background (RGB [0, 1, 0]). Determining
these bounds may take some trial and error, but the following bounds may
work for certain common background colors: \itemize{ \item Black:
lower=c(0, 0, 0); upper=c(0.1, 0.1, 0.1) \item White: lower=c(0.8, 0.8,
0.8); upper=c(1, 1, 1) \item Green: lower=c(0, 0.55, 0); upper=c(0.24, 1,
0.24) \item Blue: lower=c(0, 0, 0.55); upper=c(0.24, 0.24, 1) } If no
background filtering is needed, set bounds to some non-numeric value
(\code{NULL}, \code{FALSE}, \code{"off"}, etc); any non-numeric value is
interpreted as \code{NULL}.}

\item{title}{String for what the title the plot if plotting is on; defaults
to the image name.}

\item{a.bounds, b.bounds}{Numeric ranges for the a (green-red) and b
(blue-yellow) channels of Lab color space. Technically, a and b have
infinite range, but in practice nearly all values fall between -128 and 127
(the default). Many images will have an even narrower range than this,
depending on the lighting conditions and conversion; setting narrower
ranges will result in finer-scale binning, without generating empty bins at
the edges of the channels.}

\item{...}{Additional arguments passed to \code{\link[graphics]{barplot}}.}
}
\value{
A vector or dataframe (depending on whether \code{as.vec = TRUE}) of bin
  sizes and color coordinates.
}
\description{
Computes a histogram in CIE Lab color space by sorting pixels into specified bins.
}
\details{
\code{getLabHist} uses \code{\link{convertColorSpace}} to convert
  pixels into CIE Lab coordinates, which requires a references white. There
  are seven CIE standardized illuminants available in \code{colordistance}
  (A, B, C, E, and D50, D55, and D65), but the most common are: \itemize{
  \item \code{"A"}: Standard incandescent lightbulb \item \code{"D65"}:
  Average daylight \item \code{"D50"}: Direct sunlight}
  
Color conversions will be highly dependent on the reference white used, which
is why no default is provided. Users should look into
\href{https://en.wikipedia.org/wiki/Standard_illuminant}{standard
illuminants} to choose an appropriate reference for a dataset.

The conversion from RGB to a standardized color space (XYZ, Lab, or Luv) is
approximate, non-linear, and relatively time-consuming. Converting a large
number of pixels can be computationally expensive, so
\code{convertColorSpace} will randomly sample a specified number of rows to
reduce the time. The default sample size, 10,000 rows, takes about 1 second to
convert from sRGB to Lab space on an early 2015 Macbook with 8 GB of RAM.
Time scales about linearly with number of rows converted.

Unlike RGB or HSV color spaces, the three channels of CIE Lab color space do
not all range between 0 and 1; instead, L (luminance) is always between 0 and
100, and the a (green-red) and b (blue-yellow) channels generally vary
between -128 and 127, but usually occupy a narrower range depending on the
reference white. To achieve the best results, ranges for a and b should be
restricted to avoid generating empty bins.
}
\examples{
path <- system.file("extdata", "Heliconius/Heliconius_B/Heliconius_07.jpeg",
package="colordistance")
getLabHist(path, ref.white = "D65", bins = c(2, 3, 3), lower = rep(0.8, 3),
upper = rep(1, 3), sample.size = 1000, ylim = c(0, 1))

}
