% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/degross.R
\name{degross}
\alias{degross}
\title{Density estimation from tabulated data with given frequencies and group central moments.}
\usage{
degross(degross.data,
       phi0 = NULL, tau0 = 1000,
       use.moments = rep(TRUE,4), freq.min = 20, diag.only=FALSE,
       penalize = TRUE,
       aa = 2, bb = 1e-06, pen.order = 3, fixed.tau = FALSE,
       plotting = FALSE, verbose = FALSE, iterlim=20)
}
\arguments{
\item{degross.data}{A \link{degrossData.object} generated by \link{degrossData}.}

\item{phi0}{Starting value for the \code{K}-vector \eqn{\phi} of B-spline parameters specifying the log-density. Default: NULL.}

\item{tau0}{Starting value for the roughness penalty parameter. Default: 1000.}

\item{use.moments}{Vector with 4 logicals indicating which tabulated sample moments to use as soft constraints. Defaults: \code{rep(TRUE,4)}.}

\item{freq.min}{Minimal big bin frequency required to use the corresponding observed moments as soft constraints. Default: \code{20}.}

\item{diag.only}{Logical indicating whether to ignore the off-diagonal elements of the variance-covariance matrix of the sample central moments. Default: FALSE.}

\item{penalize}{Logical indicating whether a roughness penalty of order \code{pen.order} is required (with \eqn{\tau \sim G(aa,bb)}). Default: \code{TRUE}.}

\item{aa}{Positive real giving the first parameter in the Gamma prior for \code{tau}. Default: \code{2}.}

\item{bb}{Positive real giving the second parameter in the Gamma prior for \code{tau}. Default: \code{1e-6}.}

\item{pen.order}{Integer giving the order of the roughness penalty. Default: \code{3}.}

\item{fixed.tau}{Logical indicating whether the roughness penalty parameter \code{tau} is fixed. Default: FALSE, implying its estimation.}

\item{plotting}{Logical indicating whether an histogram of the data with the estimated density should be plotted. Default: FALSE.}

\item{verbose}{Logical indicating whether details on the estimation progress should be displayed. Default: FALSE.}

\item{iterlim}{Maximum number of iterations during the M-step. Default: 20.}
}
\value{
An object of class \code{degross} containing several components from the density estimation procedure. Details can be found in \code{\link{degross.object}}. A summary of its content can be printed using \code{\link{print.degross}} or plotted using \code{\link{plot.degross}}.
}
\description{
{Estimation of a density from tabulated summary statistics evaluated within each of the big bins (or classes) partitioning the variable support. These statistics include class frequencies and central moments of orders one up to four. The log-density is modelled using a linear combination of penalized B-splines. The multinomial log-likelihood involving the frequencies adds up to a roughness penalty based on differences of neighboring B-spline coefficients and to the log of a root-n approximation of the sampling density of the observed vector of central moments within each class. The so-obtained penalized log-likelihood is maximized using the EM algorithm to get an estimation of the spline parameters and, hence, of the variable density and related quantities such as quantiles, see Lambert (2021) for details.
}
}
\examples{
## Simulate grouped data
sim = simDegrossData(n=3500, plotting=TRUE,choice=2,J=3)
print(sim$true.density) ## Display density of the data generating mechanism

## Create a degrossData object
obj.data = with(sim, degrossData(Big.bins=Big.bins, freq.j=freq.j, m.j=m.j))
print(obj.data)

## Estimate the density underlying the grouped data
obj.fit = degross(obj.data)

## Plot the estimated density...
plot(obj.fit)
## ... and compare it with the ('target') density used to simulate the data
curve(sim$true.density(x),add=TRUE,col="red",lwd=2)
legend("topleft",
       legend=c("Observed freq.","Target density","Estimated density"),
       col=c("grey85","red","black"), lwd=c(10,2,2),
       lty=c("solid","solid","dashed"), box.lty=0, inset=.02)

}
\references{
Lambert, P. (2021) Moment-based density and risk estimation from grouped summary statistics. arXiv:2107.03883.
}
\seealso{
\code{\link{degross.object}}, \code{\link{ddegross}}, \code{\link{pdegross}}, \code{\link{qdegross}}.
}
\author{
Philippe Lambert \email{p.lambert@uliege.be}
}
