% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rpv_plot.R
\name{rpv_plot}
\alias{rpv_plot}
\title{Racially Polarized Voting Analysis (RPV) Plot}
\usage{
rpv_plot(
  rpvDF = NULL,
  title = "Racially Polarized Voting Analysis Results",
  subtitle = "Estimated Vote for Candidates by Race",
  legend_name = "Voters' Race:",
  voter_races = NULL,
  colors = NULL,
  position_dodge_width = 0.8,
  bar_size = NULL,
  label_size = 4,
  contest_name_size = 20,
  cand_name_size = 6,
  contest_name_pad = NULL,
  cand_name_pad = -1.5,
  contest_sep = NULL,
  shade_col = "grey75",
  shade_alpha = 0.1,
  panel_spacing = NULL,
  breaks = seq(0, 100, 20),
  lims = c(0, 110),
  includeErrorBand = FALSE,
  includeCandName = TRUE,
  panelBy = NULL,
  txtInBar = NULL
)
}
\arguments{
\item{rpvDF}{A data.frame containing RPV results}

\item{title}{The plot title}

\item{subtitle}{The plot subtitle}

\item{legend_name}{The legend title}

\item{voter_races}{A vector of the unique voter races contained in the 
\code{Voter_Race} column of \code{rpvDF}. This argument will set the order 
in which voter races are displayed in the plot and legend. Can be used with 
\code{colors}, to indicate the which color of the plot to associate with each  
voter race.}

\item{colors}{Defines the plot colors for the voter race groups. Colors must 
be listed in the desired order with respect \code{voter_races} if arguments 
are used together.}

\item{position_dodge_width}{The width value indicating  spacing between the 
plot bars. Passed to \code{position_dodge()}.}

\item{bar_size}{The size of plot bars. Passed to \code{geom_linerange()}.}

\item{label_size}{The size of RPV estimate label}

\item{contest_name_size}{Text size of contest name}

\item{cand_name_size}{Text size of candidate names if 
\code{includeCandName = TRUE}}

\item{contest_name_pad}{Padding between contest name and y axis}

\item{cand_name_pad}{Padding between candidate name and y axis if 
\code{includeCandName = TRUE}.}

\item{contest_sep}{String indicating how to separate contest. Options "s", 
"shade", or "shading" shade the background of every other contest. 
Options "l", "line", "lines" create light grey double lines between contests.}

\item{shade_col}{color to shade contest separation bands when
\code{contest_sep = "s"}. Defaults to light grey.}

\item{shade_alpha}{alpha parameter passed to \code{geom_tile()} to indicate 
transparency of contest separation bands when \code{contest_sep = "s"}}

\item{panel_spacing}{Space between facet grid panels}

\item{breaks}{Numeric vector containing x axis breaks}

\item{lims}{Numeric vector containing x axis limits}

\item{includeErrorBand}{Logical indicating if the confidence interval band 
should appear on the plot. If \code{TRUE}, the RPV estimate labels will
appear in the middle of each bar instead of at the ends so they don't cover 
the error bands.}

\item{includeCandName}{Logical indicating if candidate names should appear on
the left side of the plot.}

\item{panelBy}{Column name from rpvDF passed to \code{facet_grid()} to create panels.}

\item{txtInBar}{Logical indicating location of the RPV estimate labels. If, TRUE,
estimates will be in the middle of the plot bars. If FALSE, they will be at the end
of the bars.}
}
\value{
Bar plot visualization of RPV analysis as a ggplot2 object
}
\description{
Creates a custom visualization of RPV results
}
\examples{
\donttest{
library(eiExpand)
data(example_rpvDF)

# Note that these plots are designed to be 
# saved as a png using ggplot2::ggsave(). See first example for recommending 
# sizing, noting that height and weight arguments may need adjusting 
# depending on plot attributes such as number of contests and paneling 

# plot county-level results with all defaults
rpvDF_county <- example_rpvDF \%>\% dplyr::filter(Jurisdiction == "County")
rpv_plot(rpvDF_county)

# save to png with recommended sizing
# ggplot2::ggsave("rpv_plot_default.png", height = 10, width = 15)

# include CI bands
rpv_plot(rpvDF_county, includeErrorBand = TRUE)

# include CI bands with estimate labels outside bar
rpv_plot(
  rpvDF_county, 
  includeErrorBand = TRUE,
  txtInBar = FALSE
)

# panel by preferred candidate
rpvDF_county$Year <- paste(rpvDF_county$Year,
                           "\n") # so contest and year are on different lines
rpvDF_county$Preferred_Candidate <- paste(rpvDF_county$Preferred_Candidate, 
                                          "\nPreferred Candidate")
rpv_plot(
  rpvDF_county,
  panel_spacing = 6,
  panelBy = "Preferred_Candidate"
)

# plot all jurisdictions with panels
rpv_plot(example_rpvDF, panelBy = "Jurisdiction")
# add contest separation shading
rpv_plot(
  example_rpvDF, 
  panelBy = "Jurisdiction",
  contest_sep = "s"
)

# plot panels by voter_race and remove legend
rpv_plot(rpvDF_county,
panel_spacing = 6, 
panelBy = "Voter_Race") + 
  ggplot2::theme(legend.position="none")
  
}
}
\author{
Rachel Carroll <rachelcarroll4@gmail.com>

Loren Collingwood <lcollingwood@unm.edu>

Kassra Oskooii  <kassrao@gmail.com>
}
