% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute_elastic_mean.R
\name{compute_elastic_mean}
\alias{compute_elastic_mean}
\title{Compute a elastic mean for a collection of curves}
\usage{
compute_elastic_mean(
  data_curves,
  knots = seq(0, 1, len = 5),
  type = c("smooth", "polygon"),
  closed = FALSE,
  eps = 0.01,
  pen_factor = 100,
  max_iter = 50
)
}
\arguments{
\item{data_curves}{list of \code{data.frame}s with observed points in each row. Each
variable is one coordinate direction. If there is a variable \code{t},
it is treated as the time parametrization, not as an additional coordinate.}

\item{knots}{set of knots for the mean spline curve}

\item{type}{if "smooth" linear srv-splines are used which results in a differentiable mean curve
if "polygon" the mean will be piecewise linear.}

\item{closed}{\code{TRUE} if the curves should be treated as closed.}

\item{eps}{the algorithm stops if L2 norm of coefficients changes less}

\item{pen_factor}{penalty factor forcing the mean to be closed}

\item{max_iter}{maximal number of iterations}
}
\value{
an object of class \code{elastic_mean}, which is a \code{list}
with entries
  \item{type}{"smooth" if mean was modeled using linear srv-splines or
  "polygon" if constant srv-splines are used}
  \item{coefs}{spline coeffiecients}
  \item{knots}{spline knots}
  \item{data_curves}{list of \code{data.frame}s with observed points in each row.
  First variable \code{t} gives the initial parametrization, second variable \code{t_optim}
  the optimal parametrization when the curve is aligned to the mean.}
  \item{closed}{\code{TRUE} if the mean is supposed to be a closed curve.}
}
\description{
Computes a Fréchet mean for the curves stored in \code{data_curves}) with respect
to the elastic distance. Constructor function for class \code{elastic_mean}.
}
\examples{
curve <- function(t){
  rbind(t*cos(13*t), t*sin(13*t))
}
set.seed(18)
data_curves <- lapply(1:4, function(i){
  m <- sample(10:15, 1)
  delta <- abs(rnorm(m, mean = 1, sd = 0.05))
  t <- cumsum(delta)/sum(delta)
  data.frame(t(curve(t)) + 0.07*t*matrix(cumsum(rnorm(2*length(delta))),
             ncol = 2))
})

#compute elastic means
knots <- seq(0,1, length = 11)
smooth_elastic_mean <- compute_elastic_mean(data_curves, knots = knots)
plot(smooth_elastic_mean)

knots <- seq(0,1, length = 15)
polygon_elastic_mean <- compute_elastic_mean(data_curves, knots = knots, type = "poly")
lines(get_evals(polygon_elastic_mean), col = "blue", lwd = 2)

#compute closed smooth mean, takes a little longer
\donttest{
knots <- seq(0,1, length = 11)
closed_elastic_mean <- compute_elastic_mean(data_curves, knots = knots, closed = TRUE)
plot(closed_elastic_mean)}
}
