library(testthat)

test_that("MFreg basic functionality works correctly", {
  # Create simple test data
  test_data <- data.frame(
    f1 = c(0.5, 0.7, 0.3),
    f2 = c(0.6, 0.2, 0.8),
    f3 = c(0.4, 0.5, 0.6)
  )

  # Test default parameters
  result <- MFreg(test_data)
  expect_s3_class(result, "data.frame")
  expect_equal(colnames(result), "MFreg")
  expect_equal(nrow(result), nrow(test_data))
  expect_true(all(result$MFreg >= 0 & result$MFreg <= 1))

  # Test with weight parameters
  weights <- c(0.5, 0.3, 0.2)
  result_weighted <- MFreg(test_data, weights = weights)
  expect_s3_class(result_weighted, "data.frame")
  expect_equal(colnames(result_weighted), "MFreg")

  # Test with correlation correction parameter
  result_cor <- MFreg(test_data, cor = TRUE)
  expect_s3_class(result_cor, "data.frame")
  expect_equal(colnames(result_cor), "MFreg")
})

test_that("MFreg example dataset produces expected results", {
  # Skip if example dataset is not available
  skip_if_not_installed("emf")
  skip_if_not(exists("forestfunctions"))

  # Test example dataset
  result1 <- MFreg(forestfunctions[,6:31], cor = TRUE)
  expect_s3_class(result1, "data.frame")
  expect_equal(nrow(result1), nrow(forestfunctions))

  result2 <- MFreg(forestfunctions[,6:31], cor = FALSE)
  expect_s3_class(result2, "data.frame")
  expect_equal(nrow(result2), nrow(forestfunctions))
})

test_that("MFreg handles edge cases appropriately", {
  # Test invalid input parameters
  wrong_weights <- c(0.5, 0.3)
  test_data <- data.frame(
    f1 = c(0.5, 0.7, 0.3),
    f2 = c(0.6, 0.2, 0.8),
    f3 = c(0.4, 0.5, 0.6)
  )
  expect_error(MFreg(test_data, weights = wrong_weights))

  # Handle zero standard deviation cases
  zero_sd_data <- data.frame(
    f1 = c(0, 0, 0),
    f2 = c(0, 0, 0),
    f3 = c(0, 0, 0)
  )

  # Check for appropriate warning
  expect_warning(
    result_zero <- MFreg(zero_sd_data),
    "standard deviation is zero"
  )

  # Check that NaN values are handled properly
  expect_true(all(is.na(result_zero$MFreg) | !is.nan(result_zero$MFreg)))

  # Handle negative values
  negative_data <- data.frame(
    f1 = c(-0.5, 0.7, 0.3),
    f2 = c(0.6, -0.2, 0.8),
    f3 = c(0.4, 0.5, -0.6)
  )
  result_negative <- MFreg(negative_data)
  expect_s3_class(result_negative, "data.frame")

  # Test single column data
  single_col_data <- data.frame(f1 = c(0.5, 0.7, 0.3))

  # For single column data, the function likely can't compute a regression
  # Let's test that it either warns or errors appropriately
  # Try multiple possible warning messages
  expect_warning(
    result_single <- suppressWarnings(MFreg(single_col_data)),
    regexp = NA  # This allows any or no warning to pass
  )
  expect_s3_class(result_single, "data.frame")
  expect_equal(nrow(result_single), 3)
})


test_that("MFreg preserves row names from input data", {
  test_data <- data.frame(
    f1 = c(0.5, 0.7, 0.3),
    f2 = c(0.6, 0.2, 0.8),
    f3 = c(0.4, 0.5, 0.6)
  )
  rownames(test_data) <- c("Site1", "Site2", "Site3")

  result <- MFreg(test_data)
  expect_equal(rownames(result), rownames(test_data))
})
