% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/findooberrors.R
\name{findOOBErrors}
\alias{findOOBErrors}
\title{Compute and locate out-of-bag prediction errors}
\usage{
findOOBErrors(forest, X.train, Y.train = NULL, n.cores = 1)
}
\arguments{
\item{forest}{The random forest object being used for prediction.}

\item{X.train}{A \code{matrix} or \code{data.frame} with the observations
that were used to train \code{forest}. Each row should be an observation,
and each column should be a predictor variable.}

\item{Y.train}{A vector of the responses of the observations that were used
to train \code{forest}. Required if \code{forest} was created using
\code{ranger}, but not if \code{forest} was created using \code{randomForest},
\code{randomForestSRC}, or \code{quantregForest}.}

\item{n.cores}{Number of cores to use (for parallel computation in \code{ranger}).}
}
\value{
A \code{data.table} with the following three columns:

  \item{tree}{The ID of the tree of the random forest}
  \item{terminal_node}{The ID of the terminal node of the tree}
  \item{node_errs}{A vector of the out-of-bag prediction errors that fall
  within the terminal node of the tree}
}
\description{
Computes each training observation's out-of-bag prediction error using the
random forest and, for each tree for which the training observation is
out of bag, finds the terminal node of the tree in which the training
observation falls.
}
\details{
This function accepts classification or regression random forests built using
the \code{randomForest}, \code{ranger}, \code{randomForestSRC}, and
\code{quantregForest} packages. When training the random forest using
\code{randomForest}, \code{ranger}, or \code{quantregForest}, \code{keep.inbag}
must be set to \code{TRUE}. When training the random forest using
\code{randomForestSRC}, \code{membership} must be set to \code{TRUE}.
}
\examples{
# load data
data(airquality)

# remove observations with missing predictor variable values
airquality <- airquality[complete.cases(airquality), ]

# get number of observations and the response column index
n <- nrow(airquality)
response.col <- 1

# split data into training and test sets
train.ind <- sample(1:n, n * 0.9, replace = FALSE)
Xtrain <- airquality[train.ind, -response.col]
Ytrain <- airquality[train.ind, response.col]
Xtest <- airquality[-train.ind, -response.col]

# fit random forest to the training data
rf <- randomForest::randomForest(Xtrain, Ytrain, nodesize = 5,
                                 ntree = 500, keep.inbag = TRUE)

# compute out-of-bag prediction errors and locate each
# training observation in the trees for which it is out
# of bag
train_nodes <- findOOBErrors(rf, Xtrain)

# estimate conditional mean squared prediction errors,
# biases, prediction intervals, and error distribution
# functions for the test observations. provide
# train_nodes to avoid recomputing that step.
output <- quantForestError(rf, Xtrain, Xtest,
                           train_nodes = train_nodes)

}
\seealso{
\code{\link{quantForestError}}
}
\author{
Benjamin Lu \code{<b.lu@berkeley.edu>}; Johanna Hardin \code{<jo.hardin@pomona.edu>}
}
