% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LocSpheReg.R
\name{LocSpheReg}
\alias{LocSpheReg}
\title{Local Fréchet Regression for Spherical Data}
\usage{
LocSpheReg(xin = NULL, yin = NULL, xout = NULL, optns = list())
}
\arguments{
\item{xin}{A vector of length n with input measurement points.}

\item{yin}{An n by m matrix holding the spherical data, of which the sum of squares of elements within each row is 1.}

\item{xout}{A vector of length k with output measurement points; Default: \code{xout = xin}.}

\item{optns}{A list of options control parameters specified by \code{list(name=value)}. See `Details'.}
}
\value{
A list containing the following components:
\item{xout}{Input \code{xout}.}
\item{yout}{A k by m matrix holding the fitted responses, of which each row is a spherical vector, corresponding to each element in \code{xout}.}
\item{xin}{Input \code{xin}.}
\item{yin}{Input \code{yin}.}
\item{optns}{A list of control options used.}
}
\description{
Local Fréchet regression for spherical data with respect to the geodesic distance.
}
\details{
Available control options are
\describe{
\item{bw}{A scalar used as the bandwidth or \code{"CV"} (default).}
\item{kernel}{A character holding the type of kernel functions for local Fréchet regression for densities; \code{"rect"}, \code{"gauss"}, \code{"epan"}, \code{"gausvar"}, \code{"quar"} - default: \code{"gauss"}.}
}
}
\examples{
\donttest{
set.seed(1)
n <- 200
# simulate the data according to the simulation in Petersen & Müller (2019)
xin <- runif(n)
err_sd <- 0.2
xout <- seq(0,1,length.out = 51)

phi_true <- acos(xin)
theta_true <- pi * xin
ytrue <- cbind(
  sin(phi_true) * cos(theta_true),
  sin(phi_true) * sin(theta_true),
  cos(phi_true)
)
basis <- list(
  b1 = cbind(
    cos(phi_true) * cos(theta_true),
    cos(phi_true) * sin(theta_true),
    -sin(phi_true)
  ),
  b2 = cbind(
    sin(theta_true),
    -cos(theta_true),
    0
  )
)
yin_tg <- basis$b1 * rnorm(n, mean = 0, sd = err_sd) + 
  basis$b2 * rnorm(n, mean = 0, sd = err_sd)
yin <- t(sapply(seq_len(n), function(i) {
  tgNorm <- sqrt(sum(yin_tg[i,]^2))
  if (tgNorm < 1e-10) {
    return(ytrue[i,])
  } else {
    return(sin(tgNorm) * yin_tg[i,] / tgNorm + cos(tgNorm) * ytrue[i,])
  }
}))

res <- LocSpheReg(xin=xin, yin=yin, xout=xout, optns = list(bw = 0.15, kernel = "epan"))
}
}
\references{
\cite{Petersen, A., & Müller, H.-G. (2019). "Fréchet regression for random objects with Euclidean predictors." The Annals of Statistics, 47(2), 691--719.}
}
