% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tar_terra_tiles.R
\name{tile_grid}
\alias{tile_grid}
\alias{tile_blocksize}
\alias{tile_n}
\title{Helper functions to create tiles}
\usage{
tile_grid(raster, ncol, nrow)

tile_blocksize(raster, n_blocks_row = 1, n_blocks_col = 1)

tile_n(raster, n)
}
\arguments{
\item{raster}{a SpatRaster object.}

\item{ncol}{integer; number of columns to split the SpatRaster into.}

\item{nrow}{integer; number of rows to split the SpatRaster into.}

\item{n_blocks_row}{integer; multiple of blocksize to include in each tile
vertically.}

\item{n_blocks_col}{integer; multiple of blocksize to include in each tile
horizontally.}

\item{n}{integer; total number of tiles to split the SpatRaster into.}
}
\value{
list of named numeric vectors with xmin, xmax, ymin, and ymax values
that can be coerced to SpatExtent objects with \code{\link[terra:ext]{terra::ext()}}.
}
\description{
Wrappers around \code{\link[terra:makeTiles]{terra::getTileExtents()}} that return a list of named numeric
vectors describing the extents of tiles rather than \code{SpatExtent} objects.
While these may have general use, they are intended primarily for supplying
to the \code{tile_fun} argument of \code{\link[=tar_terra_tiles]{tar_terra_tiles()}}.
}
\details{
\code{tile_blocksize()} creates extents using the raster's native block size (see
\code{\link[terra:readwrite]{terra::fileBlocksize()}}), which should be more memory efficient. Create
tiles with multiples of the raster's blocksize with \code{n_blocks_row} and
\code{n_blocks_col}. We strongly suggest the user explore how many tiles are
created by \code{tile_blocksize()} before creating a dynamically branched target
using this helper. Note that block size is a property of \emph{files} and does not
apply to in-memory \code{SpatRaster}s. Therefore, if you want to use this helper
in \code{\link[=tar_terra_tiles]{tar_terra_tiles()}} you may need to ensure the upstream target provided to
the \code{raster} argument is not in memory by setting \code{memory = "transient"}.

\code{tile_grid()} allows specification of a number of rows and
columns to split the raster into.  E.g. nrow = 2 and ncol = 2 would create 4
tiles (because it specifies a 2x2 matrix, which has 4 elements).

\code{tile_n()} creates (about) \code{n} tiles and prints the number of rows, columns,
and total tiles created.
}
\examples{
f <- system.file("ex/elev.tif", package="terra")
r <- terra::rast(f)
tile_grid(r, ncol = 2, nrow = 2)
tile_blocksize(r)
tile_n(r, 8)

\donttest{
#Example usage with tar_terra_tiles
list(
    tar_terra_rast(
        my_map,
        terra::rast(system.file("ex/logo.tif", package = "terra"))
    ),
    tar_terra_tiles(
        name = rast_split,
        raster = my_map,
        tile_fun = tile_blocksize,
        description = "Each tile is 1 block"
    ),
    tar_terra_tiles(
        name = rast_split_2blocks,
        raster = my_map,
        tile_fun = \(x) tile_blocksize(
          x,
          n_blocks_row = 2,
          n_blocks_col = 1
          ),
        description = "Each tile is 2 blocks tall, 1 block wide"
    ),
    tar_terra_tiles(
        name = rast_split_grid,
        raster = my_map,
        tile_fun = \(x) tile_grid(x, ncol = 2, nrow = 2),
        description = "Split into 4 tiles in a 2x2 grid"
    ),
    tar_terra_tiles(
        name = rast_split_n,
        raster = my_map,
        tile_fun = \(x) tile_n(x, n = 6),
        description = "Split into 6 tiles"
    )
)
}
}
\author{
Eric Scott
}
