% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/interval_widths.R
\name{interval_widths}
\alias{interval_widths}
\alias{pretty_widths}
\title{Nicely-spaced sets of interval widths}
\usage{
interval_widths(n, dist = dist_normal(), max = 1 - 0.1/n, precision = NULL)

pretty_widths(
  n,
  dist = dist_normal(),
  max = if (n <= 4) 0.95 else 1 - 0.1/n,
  precision = if (n <= 4) 0.05 else 0.01
)
}
\arguments{
\item{n}{<\link{numeric}> in \eqn{[0, \infty)}: Number of intervals to generate.}

\item{dist}{<\link[distributional:distributional-package]{distribution}>: Reference
distribution.}

\item{max}{<\link{numeric}> in \eqn{(0, 1)}: Maximum interval width.}

\item{precision}{<\link{numeric} | \link{NULL}>: If not \code{NULL}, a value in \eqn{(0, 1)}
giving the precision to round resulting widths to. In order to guarantee
\code{n} unique intervals are returned, widths will only be rounded if the
result does not create duplicate values.}
}
\value{
A length-\code{n} numeric vector of interval widths (masses) between
\code{0} and \code{1} (exclusive) in increasing order.
}
\description{
Create nicely-spaced sets of nested interval widths for use with (e.g.)
the \code{.width} parameter of \code{\link[=point_interval]{point_interval()}}, \code{\link[=stat_slabinterval]{stat_slabinterval()}}, or
\code{\link[=stat_lineribbon]{stat_lineribbon()}}:
\itemize{
\item \code{interval_widths(n)} creates a sequence of \code{n} interval widths
\eqn{p_1 \ldots p_n}, where \eqn{0 < p_i \le \textrm{max} < 1}, corresponding
to the masses of nested intervals that are evenly-spaced on a reference
distribution (by default a Normal distribution). This generalizes the idea
behind the default ~66\% and 95\% intervals in \code{\link[=stat_slabinterval]{stat_slabinterval()}} and
50\%, 80\%, and 95\% intervals in \code{\link[=stat_lineribbon]{stat_lineribbon()}}: when applied to a Normal
distribution, those intervals are roughly evenly-spaced and allow one to
see deviations from the reference distribution (such as excess kurtosis) when
the resulting intervals are \emph{not} evenly spaced.
\item \code{pretty_widths(n)} is a variant of \code{interval_widths()} with defaults for
\code{max} and \code{precision} that make the resulting intervals more human-readable,
for labeling purposes.
}

Intervals should be evenly-spaced on any symmetric reference distribution
when applied to data from distributions with the same shape. If \code{dist}
is not symmetric, intervals may only be approximately evenly-spaced above the
median.
}
\details{
Given the cumulative distribution function \eqn{F_\textrm{dist}(q)}
and the quantile function \eqn{F^{-1}_\textrm{dist}(p)} of \code{dist}, the
following is a sequence of \eqn{n + 1} evenly-spaced quantiles of \code{dist}
that could represent upper limits of nested intervals, where
\eqn{q_i = q_0 + i\frac{q_n - q_0}{n}}:

\deqn{\begin{array}{rcl}
q_0, \ldots, q_n &=& F^{-1}_\textrm{dist}(0.5), \ldots, F^{-1}_\textrm{dist}(0.5 + \frac{\textrm{max}}{2})
\end{array}}

\code{interval_widths(n)} returns the \code{n} interval widths corresponding to the
upper interval limits \eqn{q_1, \ldots, q_n}:

\deqn{
2\cdot\left[F_\textrm{dist}(q_1) - 0.5\right], \ldots, 2\cdot\left[F_\textrm{dist}(q_n) - 0.5\right]
}
}
\examples{
library(ggplot2)
library(distributional)

interval_widths(1)   # 0.9
# this is roughly +/- 1 SD and +/- 2 SD
interval_widths(2)   # 0.672..., 0.95
interval_widths(3)   # 0.521..., 0.844..., 0.966...

# "pretty" widths may be useful for legends with a small number of widths
pretty_widths(1)     # 0.95
pretty_widths(2)     # 0.65, 0.95
pretty_widths(3)     # 0.50, 0.80, 0.95

# larger numbers of intervals can be useful for plots
ggplot(data.frame(x = 1:20/20)) +
  aes(x, ydist = dist_normal((x * 5)^2, 1 + x * 5)) +
  stat_lineribbon(.width = pretty_widths(10))

# large numbers of intervals can be used to create gradients -- particularly
# useful if you shade ribbons according to density (not interval width)
# (this is currently experimental)
withr::with_options(list(ggdist.experimental.slab_data_in_intervals = TRUE), print(
  ggplot(data.frame(x = 1:20/20)) +
    aes(x, ydist = dist_normal((x * 5)^2, 1 + x * 5)) +
    stat_lineribbon(
      aes(fill_ramp = after_stat(ave(pdf_min, level))),
      .width = interval_widths(40),
      fill = "gray50"
    ) +
    theme_ggdist()
))
}
\seealso{
The \code{.width} argument to \code{\link[=point_interval]{point_interval()}}, \code{\link[=stat_slabinterval]{stat_slabinterval()}},
\code{\link[=stat_lineribbon]{stat_lineribbon()}}, etc.
}
