% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bin_by_date.R
\name{bin_by_date}
\alias{bin_by_date}
\title{Aggregate data by time periods}
\usage{
bin_by_date(
  x,
  dates_from,
  n = 1,
  population = 1,
  fill_gaps = FALSE,
  date_resolution = "week",
  week_start = 1,
  .groups = "drop"
)
}
\arguments{
\item{x}{Either a data frame with a date column, or a date vector.\cr
Supported date formats are \code{date} and \code{datetime} and also commonly used character strings:
\itemize{
\item ISO dates \code{"2024-03-09"}
\item Month \code{"2024-03"}
\item Week \code{"2024-W09"} or \code{"2024-W09-1"}
}}

\item{dates_from}{Column name containing the dates to bin. Used when x is a data.frame.}

\item{n}{Numeric column with case counts (or weights). Supports quoted and unquoted column names.}

\item{population}{A number or a numeric column with the population size. Used to calculate the incidence.}

\item{fill_gaps}{Logical; If \code{TRUE}, gaps in the time series will be filled with 0 cases.
Useful for ensuring complete time series without missing periods. Defaults to \code{FALSE}.}

\item{date_resolution}{Character string specifying the time unit for date aggregation.
Possible values include: \code{"hour"}, \code{"day"}, \code{"week"}, \code{"month"}, \code{"bimonth"}, \code{"season"},
\code{"quarter"}, \code{"halfyear"}, \code{"year"}. Special values:
\itemize{
\item \code{"isoweek"}: ISO week standard (week starts Monday, \code{week_start = 1})
\item \code{"epiweek"}: US CDC epiweek standard (week starts Sunday, \code{week_start = 7})
\item \code{"isoyear"}: ISO year (corresponding year of the ISO week, differs from year by 1-3 days)
\item \code{"epiyear"}: Epidemiological year (corresponding year of the epiweek, differs from year by 1-3 days)
Defaults to \code{"week"}.
}}

\item{week_start}{Integer specifying the start of the week (1 = Monday, 7 = Sunday).
Only used when \code{date_resolution} involves weeks. Defaults to 1 (Monday).
Overridden by \code{"isoweek"} (1) and \code{"epiweek"} (7) settings.}

\item{.groups}{See \code{\link[dplyr:summarise]{dplyr::summarise()}}.}
}
\value{
A data frame with the following columns:
\itemize{
\item A date column with the same name as \code{dates_from}, where values are binned to the start of the specified time period.
\item \code{n}: Count of observations (sum of weights) for each time period
\item \code{incidence}: Incidence rate calculated as \code{n / population} for each time period
\item Any existing grouping variables are preserved
}
}
\description{
Aggregates data by specified time periods (e.g., weeks, months) and calculates (weighted)
counts. Incidence rates are also calculated using the provided population numbers.\cr\cr
This function is the core date binning engine
used by \code{\link[=geom_epicurve]{geom_epicurve()}} and \code{\link[=stat_bin_date]{stat_bin_date()}} for creating epidemiological
time series visualizations.
}
\details{
The function performs several key operations:
\enumerate{
\item \strong{Date coercion}: Converts the date column to proper Date format
\item \strong{Gap filling} (optional): Generates complete temporal sequences to fill missing time periods with zeros
\item \strong{Date binning}: Rounds dates to the specified resolution using \code{\link[lubridate:round_date]{lubridate::floor_date()}}
\item \strong{Weight and population handling}: Processes count weights and population denominators
\item \strong{Aggregation}: Groups by binned dates and sums weights to get counts and incidence
}

\strong{Grouping behaviour}:
The function respects existing grouping in the input data frame.
}
\examples{
library(dplyr)

# Create sample data
outbreak_data <- data.frame(
  onset_date = as.Date("2024-12-10") + sample(0:100, 50, replace = TRUE),
  cases = sample(1:5, 50, replace = TRUE)
)

# Basic weekly binning
bin_by_date(outbreak_data, dates_from = onset_date)

# Weekly binning with case weights
bin_by_date(outbreak_data, onset_date, n = cases)

# Monthly binning
bin_by_date(outbreak_data, onset_date,
  date_resolution = "month"
)

# ISO week binning (Monday start)
bin_by_date(outbreak_data, onset_date,
  date_resolution = "isoweek"
) |>
  mutate(date_formatted = strftime(onset_date, "\%G-W\%V")) # Add correct date labels

# US CDC epiweek binning (Sunday start)
bin_by_date(outbreak_data, onset_date,
  date_resolution = "epiweek"
)

# With population data for incidence calculation
outbreak_data$population <- 10000
bin_by_date(outbreak_data, onset_date,
  n = cases,
  population = population
)
}
