% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayes.R
\name{menbayesgl}
\alias{menbayesgl}
\title{Bayes test for F1/S1 genotype frequencies using genotype likelihoods}
\usage{
menbayesgl(
  gl,
  method = c("f1", "s1"),
  p1gl = NULL,
  p2gl = NULL,
  lg = TRUE,
  beta = NULL,
  chains = 2,
  cores = 1,
  iter = 2000,
  warmup = floor(iter/2),
  ...
)
}
\arguments{
\item{gl}{A matrix of genotype log-likelihoods. The rows index the
individuals and the columns index the genotypes. So \code{gl[i,k]}
is the genotype log-likelihood for individual \code{i} at
dosage \code{k-1}. We assume the \emph{natural} log is used (base e).}

\item{method}{Should we test for F1 proportions (\code{"f1"}) or
S1 proportions (\code{"s1"})?}

\item{p1gl}{A vector of genotype log-likelihoods for parent 1.
\code{p1gl[k]} is the log-likelihood of parent 1's data given
their genotype is \code{k}.}

\item{p2gl}{A vector of genotype log-likelihoods for parent 2.
\code{p2gl[k]} is the log-likelihood of parent 2's data given
their genotype is \code{k}.}

\item{lg}{A logical. Should we return the log Bayes factor (\code{TRUE})
or the Bayes factor (\code{FALSE})?}

\item{beta}{The concentration hyperparameters of the genotype frequencies
under the alternative of no random mating. Should be length ploidy + 1.}

\item{chains}{Number of MCMC chains. Almost always 1 is enough, but we
use 2 as a default to be conservative.}

\item{cores}{Number of cores to use.}

\item{iter}{Total number of iterations.}

\item{warmup}{Number of those iterations used in the burnin.}

\item{...}{Control arguments passed to \code{\link[rstan]{sampling}()}.}
}
\description{
Uses \code{\link[updog]{get_q_array}()} from the updog R package to
calculate segregation probabilities (assuming no double reduction) and
tests that offspring genotypes follow this distribution.
}
\examples{
\dontrun{
set.seed(1)
ploidy <- 4

## Simulate under the null ----
q <- updog::get_q_array(ploidy = 4)[3, 3, ]

## See BF increases
nvec <- c(stats::rmultinom(n = 1, size = 10, prob = q))
gl <- simgl(nvec = nvec)
menbayesgl(gl = gl, method = "f1")

nvec <- c(stats::rmultinom(n = 1, size = 100, prob = q))
gl <- simgl(nvec = nvec)
menbayesgl(gl = gl, method = "f1")

nvec <- c(stats::rmultinom(n = 1, size = 1000, prob = q))
gl <- simgl(nvec = nvec)
menbayesgl(gl = gl, method = "f1")

## Simulate under the alternative ----
q <- stats::runif(ploidy + 1)
q <- q / sum(q)

## See BF decreases
nvec <- c(stats::rmultinom(n = 1, size = 10, prob = q))
gl <- simgl(nvec = nvec)
menbayesgl(gl = gl, method = "f1")

nvec <- c(stats::rmultinom(n = 1, size = 100, prob = q))
gl <- simgl(nvec = nvec)
menbayesgl(gl = gl, method = "f1")

nvec <- c(stats::rmultinom(n = 1, size = 1000, prob = q))
gl <- simgl(nvec = nvec)
menbayesgl(gl = gl, method = "f1")

}

}
\references{
\itemize{
\item{Gerard D (2023). "Bayesian tests for random mating in polyploids." \emph{Molecular Ecology Resources}, \emph{23}(8), 1812--1822. \doi{10.1111/1755-0998.13856}.}
}
}
\author{
David Gerard
}
