% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hystar_fit.R
\name{hystar_fit}
\alias{hystar_fit}
\title{Estimate the HysTAR model using conditional least squares estimation}
\usage{
hystar_fit(
  data,
  r = c(0.1, 0.9),
  d = 0L,
  p0 = 1L,
  p1 = 1L,
  p_select = "bic",
  thin = FALSE,
  tar = FALSE
)
}
\arguments{
\item{data}{a vector, matrix or data.frame containing the outcome variable
\eqn{y} in the first column and the threshold variable \eqn{z} in the second.
Other columns are ignored.
A vector, is taken to be both the outcome and control variable,
so, in that case, a self-exciting HysTAR is fitted.}

\item{r}{A vector or a matrix with search values for \eqn{\hat{r}_0, \hat{r}_1}.
Defaults to \code{c(.1, .9)}.
\itemize{
\item A vector \code{r} must contain two values \eqn{a} and \eqn{b} in \eqn{[0, 1]}.
The search space for the thresholds will be observed values of \code{z}
between its \eqn{100a\%} and \eqn{100b\%} percentiles.
\item A matrix \code{r} allows for a custom search. It must have two columns,
such that each row represents a pair \eqn{r_0 \le r_1} to test.
You can use a matrix with one row if you don't want to estimate
the thresholds. Note that the values in these matrix should be
on the scale of \code{z}.
}}

\item{d}{A numeric vector with one or more values for the search space
of the delay parameter. Defaults to 1. Typically, d is not very large, so
a reasonable search space might be 0, 1, 2, ..., 5.}

\item{p0}{A numeric vector with one or more values for the search space
of the autoregressive order of Regime 0. Defaults to 1.}

\item{p1}{Same as \code{p0}, but for regime 1. Note that it does not need to be
equal to \code{p0}.}

\item{p_select}{The information criterion that should be minimized to select
the orders \eqn{p_0} and \eqn{p_1}. Choices:
\itemize{
\item \code{"aic"} (Akaike Information Criterion)
\item \code{"aicc"} (Corrected Akaike Information Criterion)
\item \code{"bic"} (default, Bayesian Information Criterion)
}}

\item{thin}{\code{TRUE} (default) or \code{FALSE}. Only relevant when \code{r} is a vector.
\itemize{
\item If \code{TRUE} (default), the search space for the thresholds are the
\eqn{100a\%, 100(a+0.01)\%, \dots, 100b\%} percentiles of \code{z}.
This drastically reduces computation costs while keeping a reasonably large
search space for the thresholds. Note that this is a purely practical
choice with no theoretical justification.
\item If \code{FALSE}, all observed unique values of \code{z} between
the \eqn{100a\%} and \eqn{100b\%} percentiles of \code{z} will be considered.
}}

\item{tar}{\code{TRUE} or \code{FALSE} (default). Choose \code{TRUE} if you want to fit a traditional
2-regime threshold autoregressive (TAR) model. In this model,
there is only one threshold (or equivalently, a HysTAR model with \eqn{r_0 = r_1}).}
}
\value{
An object of S3 class \code{hystar_fit}, which is a \code{list} containing the following
items:
\itemize{
\item \verb{$data}. A \code{data.frame} containing
\itemize{
\item \code{y}, the outcome variable
\item \code{z}, the threshold variable
\item \code{H}, a logical vector that indicates at which time points the hysteresis
effect is happening. Note that this vector starts with \code{NA}(s), since not all
values can be predicted in the HysTAR model. See Details.
\item \code{R}, the regime indicator vector. (Also starts with \code{NA}(s).)
}
\item \verb{$residuals}. Also accessible with the \code{residuals()} S3 method.
\item \verb{$coefficients}, a vector with the estimated coefficients.
With the \code{coef()} S3 method, the coefficients are represented in a matrix.
Use the \code{confint()} method to get the confidence intervals of the estimates.
\item \verb{$delay}, a scalar with the estimate for the delay parameter.
\item \verb{$thresholds}, a vector with the estimates of the thresholds.
\item \verb{$orders}, a vector with the estimates of the orders.
\item \verb{$resvar}, a vector with the estimates of the residual variances.
\item \verb{$rss}, the minimized residual sum of squares.
\item \verb{$ic}, a vector with the aic, the corrected aic and the bic.
\item \verb{$n}, a vector with the total effective observations and the
effective obeservations in regime 0 and regime 1.
\item \verb{$eff}, a vector with the time indicators of the effective observations.
\item \verb{$equiv}, a matrix containing equivalent estimates for the delay and thresholds,
i.e., estimates that imply exactly the same regime indicator vector, and
as a result the same minimal residual sum of squares.
\item \verb{$r_search}, a vector with the \eqn{r}-values that were considered.
\item \verb{$tar}, Logical: \code{TRUE} if a TAR model was fitted.
}

Implemented generics for the \code{hystar_fit} class:
\itemize{
\item \code{plot()} plots the \code{z} variable and the \code{y} variable above one another.
Shading of the background visualizes the regimes. Thresholds are drawn as
horizontal lines in the \code{z} plot. You can provide regime_names (char vector of 2),
main (char vector of 1), xlab (char vector of 1) and ylab (char vector of 2).
\item \code{summary()}, this also provides the p-values and standard errors for the
estimates of the coefficients.
\item \code{print()} prints the estimates within the mathematical representation of the model.
Note that the scalar multiplied with \code{e[t]} is the standard deviation
of the residuals, \emph{not} the variance. See also the model definition above.
\item \code{coef()}
\item \code{confint()}
\item \code{residuals()}
\item \code{fitted()}
\item \code{nobs()}
}
}
\description{
This function allows you to estimate the parameters of the hysteretic threshold
autoregressive (HysTAR) model.
}
\details{
In regime 0, \eqn{y_{t}} is predicted by values up to \eqn{y_{t - p_0}}.
This implies that the first \eqn{p_0} time points can not be predicted. E.g.,
if \eqn{p_0 = 2}, \eqn{y_1} would miss a value from \eqn{y_{-1}}. Similarly, the
value of the delay parameter implies that the regime is unknown for the first
\eqn{d} time points. To ensure that the same data are used on all options for
\code{d}, \code{p0} and \code{p1}, the first \code{max(d, p0, p1)} observations are discarded for
estimation of the parameters.
}
\section{The HysTAR model}{
The HysTAR model is defined as:

\eqn{ y_t = \begin{cases} \phi_{00} + \phi_{01} y_{t-1} + \cdots +
  \phi_{0 p_0} y_{t-p_0} + \sigma_{0} \epsilon_{t} \quad \mathrm{if}~R_{t} = 0 \\
  \phi_{10} + \phi_{11} y_{t-1} + \cdots + \phi_{1 p_1} y_{t-p_1} + \sigma_{1} \epsilon_{t}
  \quad \mathrm{if}~R_{t} = 1, \\ \end{cases} }

with \eqn{ R_t = \begin{cases} 0 \quad \quad \mathrm{if} \, z_{t-d} \in (-\infty, r_{0}] \\
  R_{t-1} \quad \mathrm{if} \, z_{t-d} \in (r_0, r_1] \\ 1 \quad \quad \mathrm{if} \, z_{t-d}
  \in (r_1, \infty), \\ \end{cases} }

where \eqn{p_j} denotes the order of regime \eqn{j \in \{0,1\}} with
coefficients \eqn{\phi_{j0}, \dots, \phi_{j p_j \in (-1, 1)}},
\eqn{\sigma_{j}} is the standard deviation of the residuals, and \eqn{d \in
  \{0, 1, 2, \dots\}} is a delay parameter. The parameters of primary interest are
the thresholds \eqn{r_0 \le r_1}. We let \eqn{t = 0, 1, 2, ..., T}, where \eqn{T}
is the number of observations.
}

\examples{
z <- z_sim(n_t = 200, n_switches = 5, start_regime = 1)
sim <- hystar_sim(z = z, r = c(-.5, .5), d = 2, phi_R0 = c(0, .6), phi_R1 = 1)
plot(sim)
fit <- hystar_fit(sim$data)
summary(fit)
}
\references{
Li, Guodong, Bo Guan, Wai Keung Li, en Philip L. H. Yu.
‘Hysteretic Autoregressive Time Series Models’. Biometrika 102, nr. 3
(september 2015): 717–23.

Zhu, Ke, Philip L H Yu, en Wai Keung Li. ‘Testing for the Buffered
Autoregressive Process’. Munich Personal RePEc Archive, (november 2013).
}
\author{
Daan de Jong.
}
