% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/makeModel.R
\name{modelCriterion}
\alias{modelCriterion}
\title{Model Selection Criterion}
\usage{
modelCriterion(cr, model, ...)
}
\arguments{
\item{cr}{A character string specifying the criterion to compute.
Options are \code{"AIC"}, \code{"BIC"}, \code{"AICc"}, and \code{"HQ"}. Alternatively,
a user-defined function can be provided.}

\item{model}{An object containing the fitted model. The object should include at least:
\itemize{
  \item \code{model$model} – the actual fitted model object (e.g., from \code{lm}, \code{glm}).
  \item \code{k} – the number of estimated parameters.
  \item \code{n} – the sample size.
}}

\item{...}{Additional arguments passed to the user-defined criterion function if \code{cr} is a function.}
}
\value{
A numeric value representing the selected criterion, normalized by the sample size if one of the predefined options is used.
}
\description{
Computes a model selection criterion (AIC, BIC, AICc, or HQ) or applies a user-defined function
to evaluate a statistical model.
}
\details{
This function returns model selection criteria used to compare the quality of different models.
All criteria are defined such that \strong{lower values indicate better models} (i.e., the goal is minimization).

If you wish to compare models using a maximization approach (e.g., log-likelihood),
you can multiply the result by \code{-1}.

Note: The predefined string options (e.g., \code{"AIC"}) are \strong{not} the same as the built-in R functions \code{AIC()} or \code{BIC()}.
In particular, the values returned by this function are adjusted by dividing by the sample size \code{n}
(i.e., normalized AIC/BIC), which makes it more comparable across datasets of different sizes.

The function returns:
\itemize{
  \item \strong{"AIC"}:  \eqn{ \frac{2k - 2\ell}{n} } Akaike Information Criterion divided by \code{n}.
  \item \strong{"BIC"}:  \eqn{ \frac{\log(n) \cdot k - 2\ell}{n} } Bayesian Information Criterion divided by \code{n}.
  \item \strong{"AICc"}: \eqn{ \frac{2k(k+1)}{n - k - 1} + \frac{2k - 2\ell}{n} } Corrected Akaike Information Criterion divided by \code{n}.
  \item \strong{"HQ"}: \eqn{ \frac{2 \log(\log(n)) \cdot k - 2\ell}{n} } Hannan–Quinn Criterion divided by \code{n}.
}

where:
\itemize{
  \item \eqn{k} is the number of parameters,
  \item \eqn{n} is the sample size,
  \item \eqn{\ell} is the log-likelihood of the model.
}

If \code{cr} is a function, it is called with the fitted model and any additional arguments passed through \code{...}.
}
\examples{
model <- list(model = lm(mpg ~ wt + hp, data = mtcars), k = 3, n = nrow(mtcars))
modelCriterion(AIC, model)
#
modelCriterion("AIC", model)
modelCriterion("BIC", model)

# Using a custom criterion function
my_cr_fun <- function(mod, ...) { AIC(mod) / length(mod$model[[1]]) }
modelCriterion(my_cr_fun, model)

}
\seealso{
\code{\link{kardl}}
}
