# WARNING - Generated by {fusen} from dev/flat_utils.Rmd: do not edit by hand

#' Extract spectra names and check for uniqueness
#' 
#' Given the list of raw spectra, `get_spectra_names()` extracts the spectra names
#' using the file metadata, and warns if the associated sanitized names are not unique.
#' 
#' @param spectra_list A list of [MALDIquant::MassSpectrum] objects.
#' 
#' @return A tibble with four columns
#' * `sanitized_name`: spectra names based on `fullName` where dots and dashes are converted to underscores
#' * `name`: spectra name using the `name` label in the spectra metadata
#' * `fullName`: spectra full name using the `fullName` label in the spectra metadata
#' * `file`: the path to the raw spectra data
#' 
#' @export
#' @examples
#' # Get an example directory of six Bruker MALDI Biotyper spectra
#' directory_biotyper_spectra <- system.file(
#'   "toy-species-spectra",
#'   package = "maldipickr"
#' )
#' # Import the six spectra
#' spectra_list <- import_biotyper_spectra(directory_biotyper_spectra)
#' # Extract the names
#' get_spectra_names(spectra_list)
#'
#' # Artificially create duplicated entries to show the warning
#' get_spectra_names(spectra_list[c(1,1)])
get_spectra_names <- function(spectra_list){
    object_class <- vapply(spectra_list, base::class, FUN.VALUE = character(1)) %>%
      base::unique()
    if(object_class != "MassSpectrum"){
      stop(
        "The 'spectra_list' is not a list of MALDIquant::MassSpectrum objects!"
      )
    }
    
    # Need to make sure that the spectra are not empty here to avoid
    # a tibble issue like:
    # Error in `tibble::as_tibble_row()`:
    # ! Columns 1, 2, and 3 must be named.
    #
    # Therefore, error if the spectra is empty or not
    empty_spectra <- vapply(spectra_list, MALDIquant::isEmpty, FUN.VALUE = logical(1))
    
    if(any(empty_spectra)){
      stop(
        "Empty spectra detected! Preprocess the data accordingly using `check_spectra()`"
      )
    }
    
    spectra_names <- lapply(spectra_list,  function(spectra){
      MALDIquant::metaData(spectra)[c("name", "fullName", "file")] %>%
        tibble::as_tibble_row()
    }) %>%
      dplyr::bind_rows() %>%
      dplyr::mutate(
        "sanitized_name" = gsub("[-\\.]", "_",.data$fullName)
      ) %>% 
      dplyr::relocate("sanitized_name")
    
    if( nrow(spectra_names) > dplyr::n_distinct(spectra_names[["sanitized_name"]])){
      warning(
        "Non-unique values in spectra names!",
        "\n\nQuickfix: use `dplyr::mutate(sanitized_name = base::make.unique(sanitized_name))`"
      )
    }
      return(spectra_names)
}
