#' Value-at-Risk
#'
#' @param alpha Numeric argument for Expected-Shortfall, between 0 and 1
#' @param dist String for the type of distribution (gaussian or skew-t)
#' @param params Numeric vector containing parameters of the distribution
#'
#' @description
#' The function allows to calculate Value-at-Risk for a given distribution. It takes as parameters alpha (risk level), a distribution and the parameters associated with this distribution. For example, for a normal distribution, the user must enter the mean and the standard deviation. Currently, the function can calculate the Value-at-Risk for the normal distribution and for the skew-t distribution (Azzalini and Capitianio, 2003)
#'
#' @references Azzalini, Adelchi, and Antonella Capitanio. "Distributions generated by perturbation of symmetry with emphasis on a multivariate skew t‐distribution." Journal of the Royal Statistical Society: Series B (Statistical Methodology) 65.2 (2003): 367-389.
#' @references Azzalini, Adelchi, and Maintainer Adelchi Azzalini. "Package ‘sn’." The skew-normal and skew-t distributions (2015): 1-3.
#'
#' @return Numeric value for the Value-at-Risk given the distribution and the alpha risk
#' @export
#'
#' @examples
#' f_VaR(0.95, "gaussian", params=c(0,1))
#'
f_VaR <- function(alpha, dist, params){
  # error management and match function
  if(dist=="gaussian"){
    if (length(params)!=2){
      stop("'params' must contain 2 values")
    }else{
    }
    results<-qnorm(1 - alpha,params[1],params[2])
  }else if(dist=="skew-t"){
    if (length(params)!=4){
      stop("'params' must contain 4 values")
    }else{
    }
    results<- qst(1 - alpha,xi=params[1],omega=params[2],alpha=params[3], nu=params[4],tol = 1e-04)
  }
  return(results)
}
