% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/computeInformation.R
\name{computeThreePointInfo}
\alias{computeThreePointInfo}
\title{Compute (conditional) three-point information}
\usage{
computeThreePointInfo(
  x,
  y,
  z,
  df_conditioning = NULL,
  maxbins = NULL,
  cplx = c("nml", "bic"),
  n_eff = -1,
  sample_weights = NULL,
  is_continuous = NULL
)
}
\arguments{
\item{x}{[a vector]
The \eqn{X} vector that contains the observational data of the first variable.}

\item{y}{[a vector]
The \eqn{Y} vector that contains the observational data of the second variable.}

\item{z}{[a vector]
The \eqn{Z} vector that contains the observational data of the third variable.}

\item{df_conditioning}{[a data frame]
The data frame of the observations of the set of conditioning variables
\eqn{U}.}

\item{maxbins}{[an integer]
When the data contain continuous variables, the maximum number of bins
allowed during the discretization. A smaller number makes the computation
faster, a larger number allows finer discretization.}

\item{cplx}{[a string]
The complexity model:
\itemize{
\item["bic"] Bayesian Information Criterion
\item["nml"] Normalized Maximum Likelihood, more accurate complexity cost
compared to BIC, especially on small sample size.
}}

\item{n_eff}{[an integer]
The effective number of samples. When there is significant autocorrelation
between successive samples, you may want to specify an effective number of
samples that is lower than the total number of samples.}

\item{sample_weights}{[a vector of floats]
Individual weights for each sample, used for the same reason as the effective
number of samples but with individual weights.}

\item{is_continuous}{[a vector of booleans]
Specify if each variable is to be treated as continuous (TRUE) or discrete
(FALSE), must be of length `ncol(df_conditioning) + 3`, in the order
\eqn{X, Y, Z, U1, U2, ...}. If not specified, factors and character vectors
are considered as discrete, and numerical vectors as continuous.}
}
\value{
A list that contains :
\itemize{
\item i3: The estimation of (conditional) three-point information without the
complexity cost.
\item i3k: The estimation of (conditional) three-point information with the
complexity cost (\emph{i3k = i3 - cplx}).
\item i2: For reference, the estimation of (conditional) mutual information
\eqn{I(X;Y|U)} used in the estimation of \emph{i3}.
\item i2k: For reference, the estimation of regularized (conditional) mutual
information \eqn{Ik(X;Y|U)} used in the estimation of \emph{i3k}.
}
}
\description{
Three point information is defined and computed as the
difference of mutual information and conditional mutual information, e.g.
\deqn{I(X;Y;Z|U) = I(X;Y|U) - Ik(X;Y|U,Z)}
For discrete or categorical variables, the three-point information is
computed with the empirical frequencies minus a complexity cost
(computed as BIC or with the Normalized Maximum Likelihood).
}
\details{
For variables \eqn{X}, \eqn{Y}, \eqn{Z} and a set of conditioning
variables \eqn{U}, the conditional three point information is defined as
\deqn{Ik(X;Y;Z|U) = Ik(X;Y|U) - Ik(X;Y|U,Z)}
where \eqn{Ik} is the shifted or regularized conditional mutual information.
See \code{\link{computeMutualInfo}} for the definition of \eqn{Ik}.
}
\examples{
library(miic)
N <- 1000
# Dependence, conditional independence : X <- Z -> Y
Z <- runif(N)
X <- Z * 2 + rnorm(N, sd = 0.2)
Y <- Z * 2 + rnorm(N, sd = 0.2)
res <- computeThreePointInfo(X, Y, Z)
message("I(X;Y;Z) = ", res$i3)
message("Ik(X;Y;Z) = ", res$i3k)

\donttest{
# Independence, conditional dependence : X -> Z <- Y
X <- runif(N)
Y <- runif(N)
Z <- X + Y + rnorm(N, sd = 0.1)
res <- computeThreePointInfo(X, Y, Z)
message("I(X;Y;Z) = ", res$i3)
message("Ik(X;Y;Z) = ", res$i3k)
}
}
\references{
\itemize{
\item Cabeli \emph{et al.}, PLoS Comput. Biol. 2020, \href{https://journals.plos.org/ploscompbiol/article?id=10.1371/journal.pcbi.1007866}{Learning clinical networks from medical records based on information estimates in mixed-type data}
\item Affeldt \emph{et al.}, UAI 2015, \href{https://auai.org/uai2015/proceedings/papers/293.pdf}{Robust Reconstruction of Causal Graphical Models based on Conditional 2-point and 3-point Information}
}
}
