#' Compute Neutrosophic Interval Metrics
#'
#' Calculates various metrics for neutrosophic interval data.
#' @param data A data frame with 'Auxili_min', 'Auxili_max', 'Study_min', 'Study_max'
#' @return List of calculated metrics
#' @export
compute_all_metrics <- function(data) {
  .check_neutro_data(data)

  compute_mean_interval_X <- function(data) {
    c(mean(data$Auxili_min), mean(data$Auxili_max))
  }

  apply_subtraction_rule_X <- function(data, mean_max_X, mean_min_X) {
    data.frame(
      min_dev_X = data$Auxili_min - mean_max_X,
      max_dev_X = data$Auxili_max - mean_min_X
    )
  }

  compute_sd_X <- function(subtracted_intervals_X) {
    c(
      sqrt(mean(subtracted_intervals_X$min_dev_X^2)),
      sqrt(mean(subtracted_intervals_X$max_dev_X^2))
    )
  }

  compute_cv_X <- function(sd_min_X, sd_max_X, mean_min_X, mean_max_X) {
    c(sd_min_X / mean_max_X, sd_max_X / mean_min_X)
  }

  compute_kurtosis_X <- function(subtracted_intervals_X) {
    c(
      moments::kurtosis(subtracted_intervals_X$min_dev_X),
      moments::kurtosis(subtracted_intervals_X$max_dev_X)
    )
  }

  compute_mean_interval_Y <- function(data) {
    c(mean(data$Study_min), mean(data$Study_max))
  }

  subtraction_rule_Y <- function(data, mean_max_Y, mean_min_Y) {
    data.frame(
      min_dev_Y = data$Study_min - mean_max_Y,
      max_dev_Y = data$Study_max - mean_min_Y
    )
  }

  compute_sd_Y <- function(subtracted_intervals_Y) {
    c(
      sqrt(mean(subtracted_intervals_Y$min_dev_Y^2)),
      sqrt(mean(subtracted_intervals_Y$max_dev_Y^2))
    )
  }

  compute_cv_Y <- function(sd_min_Y, sd_max_Y, mean_min_Y, mean_max_Y) {
    c(sd_min_Y / mean_max_Y, sd_max_Y / mean_min_Y)
  }

  calculate_correlation <- function(subtracted_intervals_X, subtracted_intervals_Y) {
    c(
      stats::cor(subtracted_intervals_X$min_dev_X, subtracted_intervals_Y$min_dev_Y),
      stats::cor(subtracted_intervals_X$max_dev_X, subtracted_intervals_Y$max_dev_Y)
    )
  }

  mean_interval_X <- compute_mean_interval_X(data)
  subtracted_intervals_X <- apply_subtraction_rule_X(data, mean_interval_X[2], mean_interval_X[1])
  sd_interval_X <- compute_sd_X(subtracted_intervals_X)
  cv_interval_X <- compute_cv_X(sd_interval_X[1], sd_interval_X[2], mean_interval_X[1], mean_interval_X[2])
  kurtosis_interval_X <- compute_kurtosis_X(subtracted_intervals_X)
  mean_interval_Y <- compute_mean_interval_Y(data)
  subtracted_intervals_Y <- subtraction_rule_Y(data, mean_interval_Y[2], mean_interval_Y[1])
  sd_interval_Y <- compute_sd_Y(subtracted_intervals_Y)
  cv_interval_Y <- compute_cv_Y(sd_interval_Y[1], sd_interval_Y[2], mean_interval_Y[1], mean_interval_Y[2])
  correlation_results <- calculate_correlation(subtracted_intervals_X, subtracted_intervals_Y)

  list(
    mean_interval_X = mean_interval_X,
    sd_interval_X = sd_interval_X,
    cv_interval_X = cv_interval_X,
    kurtosis_interval_X = kurtosis_interval_X,
    mean_interval_Y = mean_interval_Y,
    sd_interval_Y = sd_interval_Y,
    cv_interval_Y = cv_interval_Y,
    correlation_results = correlation_results
  )
}
