source("R/comp.R")
source("R/aux_fun.R")
source("R/is_nn.R")

#' @title slm
#'
#' @description The function that returns the left  scalar multiplication
#' neural network
#'
#' @param a A real number.
#' @param nu A neural network of the kind created by create_nn.
#'
#' @return Returns a neural network that is \eqn{a \triangleright \nu}. This
#' instantiates as \eqn{a \cdot f(x)} under continuous function activation. More specifically
#' we define operation as:
#'
#' Let \eqn{\lambda \in \mathbb{R}}. We will denote by \eqn{(\cdot) \triangleright (\cdot):
#' \mathbb{R} \times \mathsf{NN} \rightarrow \mathsf{NN}} the function satisfying for all
#' \eqn{\nu \in \mathsf{NN}} and \eqn{\lambda \in \mathbb{R}} that \eqn{\lambda \triangleright \nu =
#' \mathsf{Aff}_{\lambda \mathbb{I}_{\mathsf{I}(\nu)},0} \bullet \nu}.

#' @references Definition 2.3.4. Jentzen, A., Kuckuck, B., and von Wurstemberger, P. (2023).
#' Mathematical introduction to deep learning: Methods, implementations,
#' and theory. \url{https://arxiv.org/abs/2310.20360}.
#'
#' \emph{Note:} We will have two versions of this operation, a prefix and an
#' infix version.
#'
#' @examples
#'
#' 5 |> slm(Prd(2.1, 0.1))
#' Prd(2.1, 0.1) |> srm(5)
#'
#' @export


slm <- function(a, nu) {
  if (a |> is.numeric() &&
    length(a) == 1 &&
    a |> is.finite() &&
    nu |> is_nn()) {
    nu |> out() -> constant_matrix_size
    list() -> multiplier_network
    a |> diag(constant_matrix_size) -> W
    0 |> matrix(constant_matrix_size) -> b
    list(W = W, b = b) -> multiplier_network[[1]]
    multiplier_network |> comp(nu) -> return_network
    return(return_network)
  } else {
    stop("a must be a real number and nu must be a neural network")
  }
}


#' @title srm
#' @description The function that returns the right scalar multiplication
#' neural network
#'
#' @param nu A neural network of the type generated by create_nn().
#' @param a A real number.
#'
#' @return Returns a neural network that is \eqn{\nu \triangleleft a}. This
#' instantiates as \eqn{f(a \cdot x)}.under continuous function activation. More
#' specifically we will define this operation as:
#'
#' Let \eqn{\lambda \in \mathbb{R}}. We will denote by \eqn{(\cdot) \triangleleft (\cdot):
#' \mathsf{NN} \times \mathbb{R} \rightarrow \mathsf{NN}} the function satisfying for all
#' \eqn{\nu \in \mathsf{NN}} and \eqn{\lambda \in \mathbb{R}} that \eqn{\nu \triangleleft \lambda =
#' \nu \bullet \mathsf{Aff}_{\lambda \mathbb{I}_{\mathsf{I}(\nu)},0}}.
#'
#' @references Definition 2.3.4. Jentzen, A., Kuckuck, B., and von Wurstemberger, P. (2023).
#' Mathematical introduction to deep learning: Methods, implementations,
#' and theory. \url{https://arxiv.org/abs/2310.20360}.
#'
#' \emph{Note:} We will have two versions of this operation, a prefix
#' and an infix version.
#' @export

srm <- function(nu, a) {
  if (a |> is.numeric() &&
    length(a) == 1 &&
    a |> is.finite() &&
    nu |> is_nn()) {
    nu |> inn() -> constant_matrix_size
    list() -> multiplier_network
    a |> diag(constant_matrix_size) -> W
    0 |> matrix(constant_matrix_size) -> b
    list(W = W, b = b) -> multiplier_network[[1]]
    nu |> comp(multiplier_network) -> return_network
    return(return_network)
  } else {
    stop("a must be a real number and nu must be a neural network")
  }
}




#'
#' @param a A real number.
#' @param nu A neural network of the type generated by create_nn().
#'
#' @rdname slm
#' @export

`%slm%` <- function(a, nu) {
  if (a |> is.numeric() &&
    length(a) == 1 &&
    a |> is.finite() &&
    nu |> is_nn()) {
    nu |> out() -> constant_matrix_size
    list() -> multiplier_network
    a |> diag(constant_matrix_size) -> W
    0 |> matrix(constant_matrix_size) -> b
    list(W = W, b = b) -> multiplier_network[[1]]
    multiplier_network |> comp(nu) -> return_network
    return(return_network)
  } else {
    stop("a must be a real number and nu must be a neural network")
  }
}



#' @param nu A neural network
#' @param a A real number.
#'
#' @rdname srm
#' @export

`%srm%` <- function(nu, a) {
  if (a |> is.numeric() &&
    length(a) == 1 &&
    a |> is.finite() &&
    nu |> is_nn()) {
    nu |> inn() -> constant_matrix_size
    list() -> multiplier_network
    a |> diag(constant_matrix_size) -> W
    0 |> matrix(constant_matrix_size) -> b
    list(W = W, b = b) -> multiplier_network[[1]]
    nu |> comp(multiplier_network) -> return_network
    return(return_network)
  } else {
    stop("a must be a real number and nu must be a neural network")
  }
}
