% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wlrt.R
\name{wlrt}
\alias{wlrt}
\title{Weighted log-rank test}
\usage{
wlrt(
  formula,
  data,
  method,
  t_star = NULL,
  s_star = NULL,
  rho = NULL,
  gamma = NULL,
  timefix = TRUE
)
}
\arguments{
\item{formula}{Formula object. The response (on the left of the \code{~} operator) must be a survival object as returned by the \code{Surv} function.
The terms (on the right of the \code{~} operator) must include the treatment arm indicator, and additionally can include strata using the \code{strata} function.}

\item{data}{Data frame containing time-to-event data.}

\item{method}{Character string specifying type of weighted log-rank test.
Either \code{"lr"} for a standard log-rank test, \code{"mw"} for a modestly-weighted log-rank test,
or \code{"fh"} for the Fleming-Harrington rho-gamma family.}

\item{t_star}{Parameter \eqn{t^*} in the modestly weighted (\code{"mw"}) test, see Details.}

\item{s_star}{Parameter \eqn{s^*} in the modestly weighted (\code{"mw"}) test, see Details.}

\item{rho}{Parameter \eqn{\rho} in the Fleming-Harrington (\code{"fh"}) test, see Details.}

\item{gamma}{Parameter \eqn{\gamma} in the Fleming-Harrington (\code{"fh"}) test, see Details.}

\item{timefix}{Deal with floating point issues (as in the survival package). Default is TRUE. May need to set FALSE for simulated data.}
}
\value{
List containing the outcome of the weighted log-rank test.
\itemize{
\item \code{u} is the test statistic U for the weighted log-rank test
\item \code{v_u} is the variance of test statistic U
\item \code{z} is the Z-score
\item \code{trt_group} indicates which of the treatment arms the test statistic U corresponds to
}

In the presence of multiple strata, the results of the test on each individual strata is returned, in addition to the combined test that was proposed by
Magirr and Jiménez (2022), see \code{vignette("weighted_log_rank_tests", package="wlrt")}.
}
\description{
This function can perform two types of weighted log-rank test,
the modestly-weighted log-rank test and the Fleming-Harrington (\eqn{\rho},\eqn{\gamma}) test, in addition to the standard log-rank test.
}
\details{
Select which of the three tests to perform using argument \code{method}.
The output is calculated as outlined in \code{vignette("weighted_log_rank_tests", package="wlrt")}.
}
\examples{
library(nphRCT)
set.seed(1)
sim_data <- sim_events_delay(
  event_model=list(
    duration_c = 36,
    duration_e = c(6,30),
    lambda_c = log(2)/9,
    lambda_e = c(log(2)/9,log(2)/18)
  ),
  recruitment_model=list(
    rec_model="power",
    rec_period = 12,
    rec_power = 1
  ),
  n_c=50,
  n_e=50,
  max_cal_t = 36
)
#example setting t_star
wlrt(formula=Surv(event_time,event_status)~group,
  data=sim_data,
  method="mw",
  t_star = 4
)
#example setting s_star
wlrt(formula=Surv(event_time,event_status)~group,
  data=sim_data,
  method="mw",
  s_star = 0.5
)
#example with 1 strata
sim_data_0 <- sim_data
sim_data_0$ecog=0
sim_data_1 <- sim_events_delay(
  event_model=list(
    duration_c = 36,
    duration_e = c(6,30),
    lambda_c = log(2)/6,
    lambda_e = c(log(2)/6,log(2)/12)
  ),
  recruitment_model=list(
    rec_model="power",
    rec_period = 12,
    rec_power = 1
  ),
  n_c=50,
  n_e=50,
  max_cal_t = 36
)
sim_data_1$ecog=1
sim_data_strata<-rbind(sim_data_0,sim_data_1)
wlrt(formula=Surv(event_time,event_status)~group+strata(ecog),
  data=sim_data_strata,
  method="mw",
  t_star = 4
)
#example with 2 strata
sim_data_strata_2<-cbind(sim_data_strata,sex=rep(c("M","F"),times=100))
wlrt(formula=Surv(event_time,event_status)~group+strata(ecog)+strata(sex),
  data=sim_data_strata_2,
  method="mw",
  t_star = 4
)
}
\references{
Magirr, D. (2021).
Non-proportional hazards in immuno-oncology: Is an old perspective needed?.
Pharmaceutical Statistics, 20(3), 512-527.
\url{doi:10.1002/pst.2091}

Magirr, D. and Burman, C.F., 2019.
Modestly weighted logrank tests.
Statistics in medicine, 38(20), 3782-3790.

Magirr, D. and Jiménez, J. (2022)
Stratified modestly-weighted log-rank tests in settings with an anticipated delayed separation of survival curves
PREPRINT at \url{https://arxiv.org/abs/2201.10445}
}
