% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MCMC_normal.R
\name{hmm_mcmc_normal}
\alias{hmm_mcmc_normal}
\title{MCMC Sampler for the Hidden Markov Model with Normal emission densities}
\usage{
hmm_mcmc_normal(
  data,
  prior_T,
  prior_means,
  prior_sd,
  iter = 600,
  warmup = floor(iter/5),
  thin = 1,
  seed = sample.int(.Machine$integer.max, 1),
  init_T = NULL,
  init_means = NULL,
  init_sd = NULL,
  print_params = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{data}{(numeric) normal data}

\item{prior_T}{(matrix) prior transition matrix}

\item{prior_means}{(numeric) prior means}

\item{prior_sd}{(numeric) a single prior standard deviation}

\item{iter}{(integer) number of MCMC iterations}

\item{warmup}{(integer) number of warmup iterations}

\item{thin}{(integer) thinning parameter. By default, \code{1}}

\item{seed}{(integer) \code{optional parameter}; seed parameter}

\item{init_T}{(matrix) \code{optional parameter}; initial transition matrix}

\item{init_means}{(numeric) \code{optional parameter}; initial means}

\item{init_sd}{(numeric) \code{optional parameter}; initial standard deviation}

\item{print_params}{(logical) \code{optional parameter}; print parameters every iteration. By default, \code{TRUE}}

\item{verbose}{(logical) \code{optional parameter}; print additional messages. By default, \code{TRUE}}
}
\value{
List with following elements:
\itemize{
  \item \code{data}: data used for simulation
  \item \code{samples}: list with samples
  \item \code{estimates}: list with various estimates
  \item \code{idx}: indices with iterations after the warmup period
  \item \code{priors}: prior parameters
  \item \code{inits}: initial parameters
  \item \code{last_iter}: list with samples from the last MCMC iteration
  \item \code{info}: list with various meta information about the object
}
}
\description{
MCMC Sampler for the Hidden Markov Model with Normal emission densities
}
\details{
Please see \href{https://static-content.springer.com/esm/art\%3A10.1186\%2Fs12859-024-05751-4/MediaObjects/12859_2024_5751_MOESM1_ESM.pdf}{supplementary information} at \doi{10.1186/s12859-024-05751-4} for more details on the algorithm.

For usage recommendations please see \url{https://github.com/LynetteCaitlin/oHMMed/blob/main/UsageRecommendations.pdf}.
}
\examples{
# Simulate normal data
N <- 2^10
true_T <- rbind(c(0.95, 0.05, 0),
                c(0.025, 0.95, 0.025),
                c(0.0, 0.05, 0.95))

true_means <- c(-5, 0, 5)
true_sd <- 1.5

simdata_full <- hmm_simulate_normal_data(L = N, 
                                         mat_T = true_T, 
                                         means = true_means,
                                         sigma = true_sd)
simdata <- simdata_full$data
hist(simdata, 
     breaks = 40, 
     probability = TRUE,  
     main = "Distribution of the simulated normal data")
lines(density(simdata), col = "red")

# Set numbers of states to be inferred
n_states_inferred <- 3

# Set priors
prior_T <- generate_random_T(n_states_inferred)
prior_means <- c(-18, -1, 12)
prior_sd <- 3

# Simmulation settings
iter <- 50
warmup <- floor(iter / 5) # 20 percent
thin <- 1
seed <- sample.int(10000, 1)
print_params <- FALSE # if TRUE then parameters are printed in each iteration
verbose <- FALSE # if TRUE then the state of the simulation is printed

# Run MCMC sampler
res <- hmm_mcmc_normal(data = simdata,
                       prior_T = prior_T,
                       prior_means = prior_means,
                       prior_sd = prior_sd,
                       iter = iter,
                       warmup = warmup,
                       seed = seed,
                       print_params = print_params,
                       verbose = verbose)
res

summary(res) # summary output can be also assigned to a variable

coef(res) # extract model estimates

# plot(res) # MCMC diagnostics
}
\references{
Claus Vogl, Mariia Karapetiants, Burçin Yıldırım, Hrönn Kjartansdóttir, Carolin Kosiol, Juraj Bergman, Michal Majka, Lynette Caitlin Mikula.
Inference of genomic landscapes using ordered Hidden Markov Models with emission densities (oHMMed).
BMC Bioinformatics 25, 151 (2024). \doi{10.1186/s12859-024-05751-4}
}
