library(MASS)


#' MCCullagh's logistic model.
#'
#' McCullah, P. (1977).  A logistic model for paired comparisons with ordered categorical data.
#' Biometrika, 64(3), 449-453.
#' @param m matrix of observed counts
#' @returns a list containing
#'    w_tilde: vector of model weights for sum of normally distributed components
#'    delta_tilde: delta parameter computed using w_tilde
#'    w_star: vector of weights for Mantel-Haenszel type numerator and denominator
#'    delta_star: delta parameter computed using w_star
#'    var: variance of delta estimate
#' @export
#' @examples
#' McCullagh_logistic_model(coal_g)
McCullagh_logistic_model <- function(m) {
  result <- McCullagh_get_statistics(m)
  N <- result$N
  n <- result$n
  r <- result$r
  p <- nrow(N)

  weights <- McCullagh_compute_logistic_weights(N, n)
  w_tilde <- weights$w_tilde
  w_star <- weights$w_star

  numer_delta_star <- 0.5
  denom_delta_star <- 0.5
  delta_tilde <- 0.0
  for (j in 1:p) {
    delta_tilde <- delta_tilde + w_tilde[j] * log((r[j] + 0.5) / (n[j] - r[j] + 0.5))
    numer_delta_star <- numer_delta_star + w_star[j] * r[j]
    denom_delta_star <- denom_delta_star + w_star[j] * (n[j] - r[j])
  }
  delta_star <- log(numer_delta_star / denom_delta_star)
  delta <- delta_star

  if (is_invertible(N)) {
    N_inverse <- solve(N)
    denom <- 1.0 / t(n) %*% N_inverse %*% n
    var <- 4.0 * (1 + 0.25 * delta^2) * denom
  } else {
    var <- 4.0 * (1 + 0.25 * delta^2) / (n %*% w_star)
  }

  list(w_tilde=w_tilde, delta_tilde=delta_tilde, w_star=w_star, delta_star=delta_star, var=var,
       z_delta_tilde=delta_tilde / sqrt(var), z_delta_star=delta_star / sqrt(var))
}


#' Computes summary statistics needed to compute estimate of delta.
#'
#' @param m matrix of observed counts
#' @returns a list containing:
#'    N: matrix of sums above and below the diagonal
#'    n: vector, size of binomial
#'    r: vector, observed sums, number of successes for binomail
McCullagh_get_statistics <- function(m) {
  k <- nrow(m)
  p <- k - 1
  N <- matrix(0.0, nrow=p, ncol=p)
  n <- vector("double", p)

  for (i in 1:p) {
    for (j in i:p) {
      for (a in 1:i) {
        for (b in (j + 1): k) {
          N[i, j] <- N[i, j] + m[a, b] + m[b, a]
        }
      }
      N[j, i] <- N[i, j]
    }
    n[i] <- N[i, i]
  }

  r <- vector("double", p)
  for (j in 1:p) {
    r[j] <- 0.0
    for (a in 1:j) {
      for (b in (j + 1):k) {
        r[j] <- r[j] + m[a, b]
      }
    }
  }
  list(N=N, n=n, r=r)
}


McCullagh_compute_logistic_weights <- function(N, n) {
  D = diag(n)
  if (is_invertible(N)) {
    N_inverse <- solve(N)
    denom <- 1.0 / t(n) %*% N_inverse %*% n
    w_tilde <- t((D %*% N_inverse %*% n) %*% denom)
    w_star <- t(N_inverse %*% n)
  } else {
    N_inverse <- MASS::ginv(N)
    w_tilde <- t(D %*% N_inverse %*% n)
    # w_tilde <- t(w_tilde / sum(w_tilde))
    w_star <- t(N_inverse %*% n)
  }

  list(w_tilde=w_tilde, w_star=w_star)
}
