% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/M_statistic.R
\name{phylosignal_M}
\alias{phylosignal_M}
\title{Measure and test phylogenetic signal with M statistic}
\usage{
phylosignal_M(
  trait_dist = NULL,
  phy = NULL,
  reps = 999,
  auto_multi2di = TRUE,
  output_M_permuted = FALSE,
  cores = 1
)
}
\arguments{
\item{trait_dist}{A distance object of class \code{matrix} or \code{dist}.
Its row and column names should match the tip labels of the phylogenetic tree (\code{phy}).
The functions \code{\link[=gower_dist]{gower_dist()}} and \code{\link[cluster:daisy]{cluster::daisy()}} can be used to calculate distances using trait data.}

\item{phy}{A phylogenetic tree of class \code{phylo}.}

\item{reps}{An integer. The number of random permutations.}

\item{auto_multi2di}{A logical switch, \code{TRUE} or \code{FALSE}. Default is \code{TRUE},
then function \code{\link[ape:multi2di]{ape::multi2di()}} in \code{ape} package will be called to make the phylogeney (tree)
be dichotomous if the tree (\code{phy}) contains some polytomies.}

\item{output_M_permuted}{A logical switch, \code{TRUE} or \code{FALSE}. Default is \code{FALSE}.
If this logical switch is set to \code{TRUE}, the returned list will include the vector
of M values obtained after random permutations.}

\item{cores}{Number of cores to be used in parallel processing.
Default is 1, indicating no parallel computation is performed.
If set to 0, parallel computation is executed using \code{parallel::detectCores() - 1} number of cores.}
}
\value{
A list object containing two components.
Component \verb{$permuted} is the vector of M values obtained after random permutation for \code{reps} times;
component \verb{$observed} is the value of M statistic obtained from the original input data.
}
\description{
\code{phylosignal_M} computes the M statistic for trait(s) and evaluates
its statistical significance through a random permutation test.
The M statistic is a unified method for detecting phylogenetic signals in continuous traits,
discrete traits, and multi-trait combinations.
Blomberg and Garland (2002) provided a widely accepted statistical definition of
the phylogenetic signal, which is the "tendency for related species to resemble
each other more than they resemble species drawn at random from the tree".
The M statistic strictly adheres to the definition of phylogenetic signal,
formulating an index and developing a method of testing in strict accordance
with the definition, instead of relying on correlation analysis or evolutionary models.
The novel method equivalently expressed the textual definition of the phylogenetic signal
as an inequality equation of the phylogenetic and trait distances and constructed the M statistic.
}
\examples{
data("turtles")
# Continuous trait
trait_df <- data.frame(M1 = turtles$traits$M1, row.names = turtles$traits$specie)
trait_dist <- gower_dist(x = trait_df)
phylosignal_M(trait_dist, turtles$phylo, reps = 99) # reps=999 better

# Nominal discrete trait
trait_df <- data.frame(B1 = turtles$traits$B1, row.names = turtles$traits$specie)
trait_dist <- gower_dist(x = trait_df, type = list(factor = 1))
phylosignal_M(trait_dist, turtles$phylo, reps = 99) # reps=999 better

# Ordinal discrete trait
trait_df <- data.frame(CS1 = turtles$traits$CS1, row.names = turtles$traits$specie)
trait_dist <- gower_dist(x = trait_df, type = list(ordered = 1))
phylosignal_M(trait_dist, turtles$phylo, reps = 99) # reps=999 better

# Multi-trait Combinations
trait_df <- data.frame(turtles$traits[, c("M1", "M2", "M3", "M4", "M5")],
                       row.names = turtles$traits$specie)
trait_dist <- gower_dist(x = trait_df, type = list(factor = c("M4", "M5")))
phylosignal_M(trait_dist, turtles$phylo, reps = 99) # reps=999 better

}
\references{
Blomberg, S.P. & Garland, T., Jr (2002) Tempo and mode in evolution: phylogenetic inertia, adaptation and comparative methods. Journal of Evolutionary Biology, 15(6): 899-910.
}
\seealso{
\code{\link[=M_stat]{M_stat()}} \code{\link[=M_rand_perm]{M_rand_perm()}}
}
