% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/qif.r
\name{qif}
\alias{qif}
\title{Function to Solve a Quadratic Inference Function Model}
\usage{
qif(formula = formula(data), id = id, data = parent.frame(),
  b = NULL, tol = 1e-08, maxiter = 1000, family = gaussian,
  corstr = "independence", invfun = "finv")
}
\arguments{
\item{formula}{a formula expression as for other regression models, of the form
\code{response ~ predictors}. See the documentation of \code{\link[stats]{lm}}
and \code{\link[stats]{formula}} for details.}

\item{id}{a vector which identifies the clusters. The length of \code{id} should be
the same as the number of observations. Data are assumed to be sorted so that
observations on a cluster are contiguous rows for all entities in the formula.}

\item{data}{an optional data frame in which to interpret the variables occurring
in the \code{formula}, along with the \code{id} variables.}

\item{b}{an initial estimate for the parameters.}

\item{tol}{the tolerance used in the fitting algorithm.}

\item{maxiter}{the maximum number of iterations.}

\item{family}{a \code{family} object: a list of functions and expressions for defining
canonical link and variance functions. Families supported in \code{qif} are \code{gaussian},
\code{binomial}, \code{poisson}, and \code{gamma}; see the \code{\link[stats]{glm}} and \code{\link[stats]{formula}} documentation. Some links
are not currently available: probit link for binomial family and log link for
gamma family.}

\item{corstr}{a character string specifying the correlation structure. The
following are permitted: \code{"independence"}, \code{"exchangeable"}, \code{"AR-1"} and \code{"unstructured"}.}

\item{invfun}{a character string specifying the matrix inverse function. The
following are permitted: \code{"finv"} and \code{"ginv"}.}
}
\value{
A list containing:

\itemize{
\item{\code{title}: }{name of qif}
\item{\code{version}: }{the current version of qif}
\item{\code{model}: }{analysis model for link function, variance function and correlation struture}
\item{\code{terms}: }{analysis model for link function, variance function and correlation struture}
\item{\code{iteration}: }{the number of iterations}
\item{\code{coefficients}: }{beta esitmates value}
\item{\code{linear.perdictors}: }{linear predictor value}
\item{\code{fitted.value}: }{fitted value of y}
\item{\code{x}: }{the perdicted matrix}
\item{\code{y}: }{the response}
\item{\code{residuals}: }{y-mu}
\item{\code{pearson.resi}: }{pearson residuals}
\item{\code{scale}: }{the scale of fitted model}
\item{\code{family}: }{the type of distribution}
\item{\code{id}: }{model fitted value}
\item{\code{max.id}: }{max number of each steps}
\item{\code{xnames}: }{the values are X name of qif}
\item{\code{statistics}: }{The qif statistics}
\item{\code{Xnames}: }{the name X matrix in qif}
\item{\code{parameter}: }{parameter estimates}
\item{\code{covariance}: }{Covariance of coefficients}
}
}
\description{
Produces an object of class "\code{qif}" which is a Quadratic Inference Function fit
of the balanced longitudinal data.
}
\details{
\code{qif} provides two options of computing matrix inverses. The default
is from Fortran math library, and the other one is generalized inverse "\code{ginv}"
given in R package \code{MASS}. You can call option "\code{ginv}" through argument "\code{invfun}"
in "\code{qif()}".
}
\note{
This R package is created by transplanting a SAS macro QIF developed
originally by Peter Song and Zhichang Jiang (2006). This is version 1.5 of
this user documentation file, revised 2019-07-02.
}
\examples{
## Marginal log-linear model for the epileptic seizures count data
## (Diggle et al., 2002, Analysis of Longitudinal Data, 2nd Ed., Oxford Press).

# Read in the epilepsy data set:
data(epil)

# Fit the QIF model:
fit <- qif(y ~ base + trt + lage + V4, id=subject, data=epil,
                                       family=poisson, corstr="AR-1")
\donttest{
# Alternately, use ginv() from package MASS
fit <- qif(y ~ base + trt + lage + V4, id=subject, data=epil,
                      family=poisson, corstr="AR-1", invfun = "ginv")
}
# Print summary of QIF fit:
summary(fit)
\donttest{
## Second example: MS study
data(exacerb)

qif_BIN_IND<-qif(exacerbation ~ treatment + time + duration + time2, id=id,
                        data=exacerb, family=binomial, corstr="independence")
qif_BIN_AR1<-qif(exacerbation ~ treatment + time + duration + time2, id=id,
                        data=exacerb, family=binomial, corstr="AR-1")
qif_BIN_CS<-qif(exacerbation ~ treatment + time + duration + time2, id=id,
                        data=exacerb, family=binomial, corstr="exchangeable")
qif_BIN_UN<-qif(exacerbation ~ treatment + time + duration + time2, id=id,
                        data=exacerb, family=binomial, corstr="unstructured")

summary(qif_BIN_CS)

qif_BIN_CS$statistics

qif_BIN_CS$covariance
}

}
\references{
Qu A, Lindsay BG, Li B. Improving generalized estimating equations using quadratic
inference functions. Biometrika 2000, 87 823-836.

Qu A, Song P X-K. Assessing robustness of generalised estimating equations and
quadratic inference functions. Biometrika 2004, 91 447-459.

Qu A, Lindsay BG. Building adaptive estimating equations when inverse of covariance
estimation is difficult. J. Roy. Statist. Soc. B 2003, 65, 127-142.
}
\seealso{
glm, lm, formula.
}
\author{
Zhichang Jiang, Alberta Health Services, and Peter X.K. Song, University
of Michigan.
}
