% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TRNG.Engine.R
\name{TRNG.Engine}
\alias{TRNG.Engine}
\alias{lagfib2plus_19937_64}
\alias{lagfib2xor_19937_64}
\alias{lagfib4plus_19937_64}
\alias{lagfib4xor_19937_64}
\alias{lcg64}
\alias{lcg64_shift}
\alias{mrg2}
\alias{mrg3}
\alias{mrg3s}
\alias{mrg4}
\alias{mrg5}
\alias{mrg5s}
\alias{mt19937}
\alias{mt19937_64}
\alias{yarn2}
\alias{yarn3}
\alias{yarn3s}
\alias{yarn4}
\alias{yarn5}
\alias{yarn5s}
\title{TRNG random number engines.}
\description{
\link[=ReferenceClasses]{Reference Classes} exposing random number engines
(pseudo-random number generators) in the TRNG C++ library. Engine objects of
a class \code{engineClass} are created as \code{r <- engineClass$new(\dots)},
and a method \code{m} is invoked as \code{x$m(\dots)}. The engine object
\code{r} can be then used for generating random variates via any of the
\code{r<dist>_trng} functions (e.g., \code{\link{runif_trng}}), specifying
the optional argument \code{engine = r}.
}
\section{Classes}{


\describe{
  \item{\emph{Parallel} random number engines}{
  \code{lcg64}, \code{lcg64_shift}, \code{mrg2}, \code{mrg3}, \code{mrg3s},
  \code{mrg4}, \code{mrg5}, \code{mrg5s}, \code{yarn2}, \code{yarn3},
  \code{yarn3s}, \code{yarn4}, \code{yarn5}, \code{yarn5s}.
  }
  \item{\emph{Conventional} random number engines}{
  \code{lagfib2plus_19937_64}, \code{lagfib2xor_19937_64},
  \code{lagfib4plus_19937_64}, \code{lagfib4xor_19937_64}, \code{mt19937_64},
  \code{mt19937}.
  }
}
}

\section{Constructors}{

\describe{
  \item{\code{$new()}}{
  Construct a random engine object using default seed and internal parameters.
  }
  \item{\code{$new(seed)}}{
  Construct a random engine object with default internal parameters using the
  provided \code{seed}.
  }
  \item{\code{$new(string)}}{
  Construct a random engine object restoring its internal state and
  parameters from a character \code{string}, falling back to \code{$new()}
  for empty strings. See method \code{$toString()}.
  }
}
}

\section{Methods}{

\describe{
  \item{\code{$seed(seed)}}{
  Use the scalar integer \code{seed} to set the engine's internal state.
  }
  \item{\code{$jump(steps)}}{
  Advance by \code{steps} the internal state of the engine. Applies to
  \emph{parallel} engines only.
  }
  \item{\code{$split(p, s)}}{
  Update the internal state and parameters of the engine for generating
  directly the \code{s}th of \code{p} subsequences, with \code{s} in
  [\code{1}, \code{p}], producing one element every \code{s} starting from
  the \code{p}th. Applies to \emph{parallel} engines only.
  }
  \item{\code{$name()}, \code{$kind()}}{
  Return the name of the random number engine (e.g., \code{"yarn2"}), also
  referred to as \code{kind} in \pkg{rTRNG} similarly to \link[=Random]{base}
  \R.
  }
  \item{\code{$toString()}}{
  Return a character representation of the engine's internal state and
  parameters.
  }
  \item{\code{$copy()}}{
  Specialization of the generic method for \link[=ReferenceClasses]{Reference
  Classes}, ensuring the underlying C++ engine object is properly copied.
  }
  \item{\code{$show()}}{
  Specialization of the generic \link[methods]{show}, displaying
  \code{$toString()} (truncated to 80 characters).
  }
  \item{\code{$.Random.seed()}}{
  Return a two-element character vector with elements \code{$kind()} and
  \code{$toString()}, suitable for use in \code{\link{TRNG.Random.seed}} (and
  by a possible function returning an engine object given a
  \code{TRNG.Random.seed}).
  }
}
}

\section{Details}{

The TRNG C++ library provides a collection of random number engines
(pseudo-random number generators). In particular, compared to
\emph{conventional} engines working in a purely sequential manner,
\emph{parallel} engines can be manipulated via \code{jump} and \code{split}
operations. Jumping allows to advance the internal state by a number of steps
without generating all intermediate states, whereas split operations allow to
generate directly a subsequence obtained by decimating the original sequence.
Please consult the TRNG C++ library documentation (see \sQuote{References})
for an introduction to the concepts and details around (parallel) random
number generation and engines, including details about the state size and
period of the TRNG generators.



Random number engines from the C++ TRNG library are exposed to \R using
\pkg{\link[Rcpp]{Rcpp}} \code{\link[Rcpp:Module-class]{Module}}s. As a consequence, the
arguments to all Constructors and Methods above are not passed by name but by
order. Moreover, arguments and return values are both defined in terms of C++
data types. Details can be displayed via the standard
\link[=ReferenceClasses]{Reference Class} documentation method \code{$help}
(e.g., \code{yarn2$help(split)}).

Most of the Methods above are simple wrappers of analogous methods in the
corresponding C++ class provided by the TRNG library. A few
differences/details are worth being mentioned.
\itemize{
  \item
    Argument \code{s} of the \code{split} method is exposed to \R according
    to \R's \code{1}-based indexing, thus in the [\code{1}, \code{p}]
    interval, whereas the TRNG C++ implementation follows C++ \code{0}-based
    indexing, thus allowing values in [\code{0}, \code{p-1}].
  \item
    Constructor \code{new(string)} and method \code{toString()} rely on
    streaming operators \code{>>} and \code{<<} available for all C++ TRNG
    classes.
  \item
    TRNG C++ random number engine objects are \emph{copy-constructible} and
    \emph{assignable}, whereas their \R counterparts in \pkg{rTRNG} are
    purely reference-based. In particular, as for any \R
    \link[=ReferenceClasses]{Reference Object}, engines are not copied upon
    assignment but via the \code{$copy()} method.
}
}

\section{Random number engines details}{

\subsection{Parallel engines}{
  \describe{
    \item{\code{lcg64}}{
      Linear congruential generator with modulus \eqn{2^{64}}{2^64}.
    }
    \item{\code{lcg64_shift}}{
      Linear congruential generator with modulus \eqn{2^{64}}{2^64} and
      bit-shift transformation.
    }
    \item{\code{mrg2}, \code{mrg3}, \code{mrg4}, \code{mrg5}}{
      Multiple recurrence generators based on a linear feedback shift
      register sequence with prime modulus \eqn{2^{31}-1}{2^31-1}.
    }
    \item{\code{mrg3s}, \code{mrg5s}}{
      Multiple recurrence generators based on a linear feedback shift
      register with Sophie-Germain prime modulus.
    }
    \item{\code{yarn2}, \code{yarn3}, \code{yarn4}, \code{yarn5}}{
      YARN generators based on the delinearization of a linear feedback shift
      register sequence with prime modulus \eqn{2^{31}-1}{2^31-1}.
    }
    \item{\code{yarn3s}, \code{yarn5s}}{
      YARN generators based on the delinearization of a linear feedback shift
      register sequence with Sophie-Germain prime modulus.
    }
  }
}

\subsection{Conventional engines}{
  \describe{
    \item{\code{lagfib2plus_19937_64}, \code{lagfib4plus_19937_64}}{
      Lagged Fibonacci generator with \code{2} or \code{4} feedback taps and
      addition.
    }
    \item{\code{lagfib2xor_19937_64}, \code{lagfib4xor_19937_64}}{
      Lagged Fibonacci generator with \code{2} or \code{4} feedback taps and
      exclusive-or operation.
    }
    \item{\code{mt19937}}{
      Mersenne-Twister generating 32 random bit.
    }
    \item{\code{mt19937_64}}{
      Mersenne-Twister generating 64 random bit.
    }
  }
}
}

\examples{
## Class yarn2 used in the examples below can be replaced by any other TRNG
## engine class (only of a parallel kind for jump and split examples).

## basic constructor with default internal state (and parameters)
r <- yarn2$new()
## show the internal parameters and state
r
## return internal parameters and state as character string
r$toString()

## seed the random number engine
r$seed(117)
r
## construct with given initial seed
s <- yarn2$new(117)
identical(s$toString(), r$toString())

## construct from string representation
s <- yarn2$new(r$toString()) # implicitly creates a copy
identical(s$toString(), r$toString())
s <- yarn2$new("") # same as yarn2$new()
identical(s$toString(), yarn2$new()$toString())
\dontrun{
  ## error if the string is not a valid representation
  s <- yarn2$new("invalid")
}

## copy vs. reference
r_ref <- r # reference to the same engine object
r_cpy <- r$copy() # copy an engine
identical(r_cpy$toString(), r$toString())
rbind(c(runif_trng(4, engine = r), runif_trng(6, engine = r_ref)),
      runif_trng(10, engine = r_cpy))

## jump (and draw from reference)
runif_trng(10, engine = r_cpy)
r_ref$jump(7) # jump 7 steps ahead
runif_trng(3, engine = r) # jump has effect on the original r

## split
r_cpy <- r$copy()
runif_trng(10, engine = r)
r_cpy$split(5, 2) # every 5th element starting from the 2nd
runif_trng(2, engine = r_cpy)

## seed, jump and split can be used in c(...) as they return NULL
r <- yarn2$new()
r_cpy <- r$copy()
r$seed(117)
runif_trng(10, engine = r)
c(r_cpy$seed(117),
  r_cpy$jump(2), runif_trng(2, engine = r_cpy),
  r_cpy$split(3,2), runif_trng(2, engine = r_cpy))

## TRNG engine name/kind
r$kind()
r$name()

## use $.Random.seed() to set the current engine (as a copy)
r$.Random.seed()
TRNG.Random.seed(r$.Random.seed())

}
\references{
Heiko Bauke, \emph{Tina's Random Number Generator Library}, Version
4.23.1, \url{https://github.com/rabauke/trng4/blob/v4.23.1/doc/trng.pdf}.
}
\seealso{
\code{\link{ReferenceClasses}}, \code{\link{TRNG.Random}}.

TRNG distributions:
\code{\link{rbinom_trng}}, \code{\link{rlnorm_trng}}, \code{\link{rnorm_trng}}, \code{\link{rpois_trng}}, \code{\link{runif_trng}}.
}
