% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/robRatio.r
\name{robRatio}
\alias{robRatio}
\title{Robust estimator for ratio models}
\usage{
robRatio(
  x1,
  y1,
  gm = "b",
  wf = "T",
  scale = "AAD",
  rt = 1,
  tp = 8,
  rp.max = 100,
  cg.rt = 0.01
)
}
\arguments{
\item{x1}{single explanatory variable (a vector)}

\item{y1}{objective variable to be imputed (a vector)}

\item{gm}{indication of gamma value as follows:  \cr
gm="a": gamma=1 \cr
gm="b": gamma=1/2    (conventional ratio model) \cr
gm="c"; gamma=0    (regression model without intercept) \cr}

\item{wf}{weight function (wf=T : Tukey, wf=H : Huber)}

\item{scale}{scale for residuals. "AAD"(default) or "MAD".}

\item{rt}{sample weight (default 1)}

\item{tp}{standardized tuning parameter. choose 4, 6 or 8. Smaller figure is
more robust (default tp=8).  See details.}

\item{rp.max}{maximum number of iteration (default: rp.max=50)}

\item{cg.rt}{convergence condition to stop iteration (default: cg1=0.001)}
}
\value{
a list with the following elements
\describe{
\item{\code{cond}}{Weight function, scale, and other arguments choosed}
\item{\code{par}}{robustly estimated ratio of y1 to x1 (beta)}
\item{\code{res}}{homoscedastic quasi-residuals}
\item{\code{wt}}{robust weights}
\item{\code{rp}}{total number of iteration}
\item{\code{s1}}{changes of the scale (AAD or MAD)}
\item{\code{efg}}{error flag. 1: acalculia (all weights become zero)  0: successful termination}
}
}
\description{
This function integrates 4 functions (RrT.aad, RrT.mad,
RrH.aad and RrH.mad) for estimating generalized ratio model.  Please note
that the values for the tuning parameter \code{tp} allowed in this function
is standardized. See the vignette for the detail.
}
\examples{
require(robRatio)

x1 <- seq(1, 10, by=0.1)
#e <- rnorm(length(x1))
e <- rt(length(x1), df=3)   # error term following t distribution

b <- 2		# true value of slope

y1 <- b*x1 + x1*e			# example 1: gamma=1
y2 <- b*x1 + sqrt(x1)*e   # example 2: gamma=1/2

o1 <- robRatio(x1, y1, gm="a")
o2 <- robRatio(x1, y2, gm="b")

o1$par;  o2$par     # estimated slope

cols = RColorBrewer::brewer.pal(11, "PiYG")
cl1 <- round((o1$wt)*10+1)
cl2 <- round((o2$wt)*10+1)

oldpar <- par(mfrow=c(1,2))
plot(x1, y1, col=cols[cl1], pch=20)
plot(x1, y2, col=cols[cl2], pch=20)
par(oldpar)

}
