% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rpc.R
\name{rpc}
\alias{rpc}
\title{Ridge Partial Correlations}
\usage{
rpc(X, y, lambda = 1, XXt = NULL, ncores = 1, RAM = 6)
}
\arguments{
\item{X}{The input matrix of type `matrix' or `dgCMatrix'. No
need to center or scale X as that would be done implicitly.}

\item{y}{vector containing the response variable.}

\item{lambda}{the regularization parameter.}

\item{XXt}{the matrix UU' where U = scale(X), will be computed
if not provided. When the matrix X is large, recomputing XXt can be costly,
for example if the rpc's for a new lambda value is required. It is thus
wise to pass XXt (see function \code{XXt.compute}).}

\item{ncores}{the number of cores to be used. Default is 1.}

\item{RAM}{The algorithm will use a maximum of this much additional RAM. 
Default is 6GB. Increasing this number, within the limit of the machine,
will allow the algorithm to run faster.}
}
\value{
an object containing the following items: \cr
         - rpc: vector of rpc coefficients. \cr
         - X: the original matrix of covariates. \cr
         - XXt: the matrix UU', where U is the centered-scaled X. \cr
         - y: vector of centered and scaled response variable.\cr
         - lambda: the original lambda used.
}
\description{
The sample ridge partial correlations (RPC) can be used for a simple, efficient, and scalable variables screening method 
in ultra-high dimensional linear regression. 
These coefficients incorporate both the ridge regularized estimates of the regression coefficients
and the ridge regularized partial variances of the predictor variables, 
providing the screening method with sure screening property 
without strong assumptions on the marginal correlations. 
For more details, please see Wang et al. (2025).

This function computes the ridge partial correlations for each variable in the covariate matrix.
}
\details{
Consider the linear regression model:
\deqn{y = \beta_0 + X\beta + \epsilon,}
where \eqn{X} is the \eqn{n \times p} design matrix, \eqn{\beta} is a p-dimensional vector, 
and \eqn{\epsilon} is the n-dimensional vector of iid residual errors with mean 0 and variance \eqn{\sigma^2}, \eqn{\epsilon} is independent of \eqn{X}.

Assuming the design matrix \eqn{X} is centered and scaled, and letting \eqn{\tilde{Y} = Y - \bar{Y}_n},
the (sample) partial correlation between the \eqn{i^{th}} predictor and \eqn{Y} given the remaining predictors is 
\eqn{-P_{i+1, 1} / \sqrt{P_{1, 1}P_{i+1, i+1}}}, where \eqn{P} is the \eqn{(p+1) \times (p+1)} joint precision matrix of the response and the covariates.
In cases when \eqn{p > n}, \eqn{P} is not invertible. 
Hence, we consider the ridge regularized version of \eqn{P}, 
given by \eqn{R}, where:
\deqn{R := (n-1)\begin{bmatrix}\tilde{Y}^T\tilde{Y} & \tilde{Y}^TX\\ X^T\tilde{Y} & X^TX + \lambda I_p \end{bmatrix}^{-1}.}

The ridge partial correlations between \eqn{y} and \eqn{X_{i}} is then defined in terms of elements of \eqn{R} as follows:
\deqn{\rho_{i, \lambda} = - \dfrac{r_{i+1,1}}{\sqrt{r_{1,1} r_{i+1,i+1}}}}

where \eqn{r_{i,j}} is the \eqn{(i.j)}th entry in \eqn{R}.

For variable screening, one may choose the \eqn{K} variables with largest absolute RPC values,
for some \eqn{K > 0} (e.g., \eqn{K = n} or \eqn{n/\log p}). Alternatively, the extended BIC
criteria may be used. See \code{eBIC} function in this package.

\bold{N.B.} Further hyper-threading speed-up can be obtained by
loading the \code{MatrixExtra} package.
}
\examples{
## Toy example:
n <- 50; p <- 400;
trueidx <- 1:3 ## First three variables are important
truebeta <- c(4,5,6)
X <- matrix(rnorm(n*p), n, p) ## n x p covariate matrix
y <- 0.5 + X[,trueidx] \%*\% truebeta + rnorm(n) ## Response
res <- rpc(X,y, lambda = 0.1, ncores = 1)
order(abs(res$rpc),decreasing = TRUE)[1:10] # Top 10 variables
## Run another case with the same X and y, but pass XXt to make it faster
res2 <- rpc(X,y, lambda = 0.001,XXt = res$XXt , ncores = 1)
order(abs(res2$rpc),decreasing = TRUE)[1:10] # Top 10 variables

}
\seealso{
[eBIC()] for model selecting, [XXt.compute()] for computing crossproduct.
}
\author{
Somak Dutta

An Nguyen

Maintainer: Somak Dutta <somakd@iastate.edu>
}
