% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/safeframe-package.R
\docType{package}
\name{safeframe-package}
\alias{safeframe-package}
\alias{safeframe}
\title{Base Tools for Tagging and Validating Data}
\description{
The \pkg{safeframe} package provides tools to help tag and validate data.
The 'safeframe' class adds column level attributes to a 'data.frame'.
Once tagged, variables can be seamlessly used in downstream analyses,
making data pipelines more robust and reliable.
}
\note{
The package does not aim to have complete integration with \pkg{dplyr}
functions. For example, \code{\link[dplyr:mutate]{dplyr::mutate()}} and \code{\link[dplyr:bind_rows]{dplyr::bind_rows()}} will
not preserve tags in all cases. We only provide compatibility for
\code{\link[dplyr:rename]{dplyr::rename()}}.
}
\section{Main functions}{

\itemize{
\item \code{\link[=make_safeframe]{make_safeframe()}}: to create \code{safeframe} objects from a \code{data.frame} or
a \code{tibble}
\item \code{\link[=set_tags]{set_tags()}}: to change or add tagged variables in a \code{safeframe}
\item \code{\link[=tags]{tags()}}: to get the list of tags of a \code{safeframe}
\item \code{\link[=tags_df]{tags_df()}}: to get a \code{data.frame} of all tagged variables
\item \code{\link[=lost_tags_action]{lost_tags_action()}}: to change the behaviour of actions where tagged
variables are lost (e.g removing columns storing tagged variables) to
issue warnings, errors, or do nothing
\item \code{\link[=get_lost_tags_action]{get_lost_tags_action()}}: to check the current behaviour of actions
where tagged variables are lost
}
}

\section{Dedicated methods}{


Specific methods commonly used to handle \code{data.frame} are provided for
\code{safeframe} objects, typically to help flag or prevent actions which could
alter or lose tagged variables (and may thus break downstream data
pipelines).
\itemize{
\item \verb{names() <-} (and related functions, such as \code{\link[dplyr:rename]{dplyr::rename()}}) will
rename variables and carry forward the existing tags
\item \verb{x[...] <-} and \verb{x[[...]] <-} (see \link{sub_safeframe}): will adopt the
desired behaviour when tagged variables are lost
\item \code{print()}: prints info about the \code{safeframe} in addition to the
\code{data.frame} or \code{tibble}
}
}

\examples{

# using base R style
x <- make_safeframe(cars[1:50, ],
  mph = "speed",
  distance = "dist"
)
x

## check tagged variables
tags(x)

## robust renaming
names(x)[1] <- "identifier"
x

## example of dropping tags by mistake - default: warning
x[, 2]

## to silence warnings when tags are dropped
lost_tags_action("none")
x[, 2]

## to trigger errors when tags are dropped
# lost_tags_action("error")
# x[, 1]

## reset default behaviour
lost_tags_action()

# using tidyverse style

## example of creating a safeframe, adding a new variable, and adding a tag
## for it

if (require(dplyr) && require(magrittr)) {
  x <- cars \%>\%
    tibble() \%>\%
    make_safeframe(
      mph = "speed",
      distance = "dist"
    ) \%>\%
    mutate(result = if_else(speed > 50, "fast", "slow")) \%>\%
    set_tags(ticket = "result")

  head(x)

  ## extract tagged variables
  x \%>\%
    select(has_tag(c("ticket")))

  ## Retrieve all tags
  x \%>\%
    tags()

  ## Select based on variable name
  x \%>\%
    select(starts_with("speed"))
}

}
\seealso{
Useful links:
\itemize{
  \item \url{https://epiverse-trace.github.io/safeframe/}
  \item \url{https://github.com/epiverse-trace/safeframe}
  \item Report bugs at \url{https://github.com/epiverse-trace/safeframe/issues}
}

}
\author{
\strong{Maintainer}: Chris Hartgerink \email{chris@data.org} (\href{https://orcid.org/0000-0003-1050-6809}{ORCID})

Other contributors:
\itemize{
  \item Hugo Gruson \email{hugo@data.org} (\href{https://orcid.org/0000-0002-4094-1476}{ORCID}) [reviewer]
  \item data.org [copyright holder]
}

}
\keyword{internal}
