% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions.R
\name{scqe}
\alias{scqe}
\title{Stability controlled quasi-experiment (scqe)}
\usage{
scqe(
  post,
  treatment,
  outcome,
  min_outcome,
  max_outcome,
  delta,
  min_delta,
  max_delta,
  cohort,
  untr_pre,
  untr_post,
  tr_post,
  tr_pre,
  Y_tr_post,
  Y_untr_post,
  Y_tr_pre,
  Y_untr_pre,
  untr,
  tr,
  Y_tr,
  Y_untr,
  alpha = 0.05,
  ...
)
}
\arguments{
\item{post}{Binary vector corresponding to T = 0, 1 for each observation.}

\item{treatment}{Binary or continuous vector corresponding (usually) to [0,1]
(no treatment or treatment) for each observation.}

\item{outcome}{Continuous vector representing the outcome for each
observation.}

\item{min_outcome}{Minimum value for the outcome.
Optional, not used if \code{outcome} is supplied.}

\item{max_outcome}{Maximum value for the outcome.
Optional, not used if \code{outcome} is supplied.}

\item{delta}{Single value or vector of possible values for change in
average non-treatment outcome between cohorts (if applicable).}

\item{min_delta}{Minimum delta. Optional, not used if \code{delta} is supplied.}

\item{max_delta}{Maximum delta. Optional, not used if \code{delta} is supplied.}

\item{cohort}{Numeric, 1 or 2 depending on cohort membership.}

\item{untr_pre}{Integer number of untreated patients in the first cohort if
applicable (summary statistics input) (T=0).}

\item{untr_post}{Integer number of untreated patients in the second cohort
if applicable (summary statistics input) (T=1).}

\item{tr_post}{Integer number of treated patients in the second cohort if
applicable (summary statistics input) (T=1).}

\item{tr_pre}{Integer number of treated patients in the first cohort if
applicable (summary statistics input) (T=0).}

\item{Y_tr_post}{Outcome for patients who received treatment at time T=1
(summary statistics input).}

\item{Y_untr_post}{Outcome for patients who did not receive treatment at
time T=1 (summary statistics input).}

\item{Y_tr_pre}{Outcome for patients who did receive treatment at time T=0
(summary statistics input).}

\item{Y_untr_pre}{Outcome for patients who did not receive treatment at time
T=0 (summary statistics input).}

\item{untr}{Integer number of untreated patients (summary statistics input).}

\item{tr}{Integer number of treated patients (summary statistics input).}

\item{Y_tr}{Outcome for treated patients (summary statistics input).}

\item{Y_untr}{Outcome for untreated patients (summary statistics input).}

\item{alpha}{Numeric alpha for confidence interval (default is alpha = 0.05).}

\item{...}{Extra optional arguments.}
}
\value{
scqe object, results table
}
\description{
Main scqe function. Computes scqe estimates and corresponding confidence
intervals.
}
\examples{
set.seed(1234)
post = c(rep(0,100), rep(1,100))
tx = c(rep(0, 100), rbinom(n = 100, prob = 0.27, size = 1))
y = rbinom(n = 200, prob = 0.1 + .02 * post - 0.05 * tx, size = 1)

# Two cohorts, full data
scqe.2cohort.full = scqe(post = post, treatment = tx, outcome = y,
                        delta = seq(from = -0.1, to = 0.1, by = 0.05))
plot(scqe.2cohort.full)
summary(scqe.2cohort.full)

# One cohort, full data
scqe.1cohort.full = scqe(treatment = tx, outcome = y,
                        delta=seq(from = -0.1, to = 0.1, by = 0.05))
plot(scqe.1cohort.full)
summary(scqe.1cohort.full)

# Two cohorts, summary data only
scqe.2cohort.sum = scqe(untr_pre = 200,untr_post = 150, tr_post = 50,
                       tr_pre = 0, Y_tr_post = 20, Y_untr_post = 1,
                       Y_tr_pre = 0, Y_untr_pre = 5, min_delta = 0.1,
                       max_delta = 1)
plot(scqe.2cohort.sum)
summary(scqe.2cohort.sum)

# One cohort, summary data only
scqe.1cohort.sum = scqe(untr = 100, tr = 200, Y_untr = 5, Y_tr = 50,
                        min_delta= 0.1, max_delta = 1)
plot(scqe.1cohort.sum)
summary(scqe.1cohort.sum)

}
\references{
Hazlett, C. (2019), 'Estimating causal effects of new treatments
despite self-selection: The case of experimental medical treatments.'
Journal of Causal Inference.
}
