% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shrinkDTVPVAR.R
\name{shrinkDTVPVAR}
\alias{shrinkDTVPVAR}
\title{Markov Chain Monte Carlo (MCMC) for TVP-VAR-SV models under dynamic shrinkage priors}
\usage{
shrinkDTVPVAR(
  y,
  p = 1,
  mod_type = "double",
  const = TRUE,
  niter = 5000,
  nburn = round(niter/2),
  nthin = 1,
  display_progress = TRUE,
  TVP_params_beta = list(),
  TVP_params_sigma = list()
)
}
\arguments{
\item{y}{matrix or data frame containing the time series data. The rows correspond to the time
points and the columns to the variables.}

\item{p}{positive integer indicating the number of lags in the VAR model. The default value is 1.}

\item{mod_type}{character string that reads either \code{"triple"}, \code{"double"} or \code{"ridge"}.
Determines whether the triple gamma, double gamma or ridge prior are used for \code{theta_sr} and \code{beta_mean}.
The default is "double".}

\item{const}{logical value indicating whether a constant should be included in the model. The default value is \code{TRUE}.}

\item{niter}{positive integer, indicating the number of MCMC iterations
to perform, including the burn-in. Has to be larger than or equal to \code{nburn} + 2. The default value is 5000.}

\item{nburn}{non-negative integer, indicating the number of iterations discarded
as burn-in. Has to be smaller than or equal to \code{niter} - 2. The default value is \code{round(niter/2)}.}

\item{nthin}{positive integer, indicating the degree of thinning to be performed. Every \code{nthin} draw is kept and returned.
The default value is 1, implying that every draw is kept.}

\item{display_progress}{logical value indicating whether the progress bar and other informative output should be
displayed. The default value is \code{TRUE}.}

\item{TVP_params_beta}{\emph{optional} named list containing hyperparameter values for the TVP prior of the beta_mean matrix.
Not all have to be supplied, with those missing being replaced by the default values. Any list elements that are misnamed will be ignored and a warning will be thrown.
Can either be a list of depth 1, which results in all equations having the same hyperparameters, or a list of lists of length \code{m}, where each sub-list contains the
hyperparameters for the respective equation. If sub-lists are provided, they can be unnamed, in which case the order of the elements is assumed to be the same as the order of the equations in the data.
Alternatively, they can be named \code{eq1}, \code{eq2}, ..., \code{eqm}.
The meaning of the hyperparameters is the same as in \code{\link[shrinkTVP]{shrinkDTVP}}, as this function is used to sample the TVP coefficients.
A key difference lies in how the adaptive MH arguments for theta and beta_mean are provided. In this function, the adaptive MH arguments are provided as vectors of length 4,
where the elements control the MH steps in following order: \code{a_xi}, \code{a_tau}, \code{c_xi}, \code{c_tau}.
E.g. if \code{adaptive = c(FALSE, TRUE, TRUE, FALSE)}, then the MH step for \code{a_xi} and \code{c_tau} are non-adaptive, while the MH step for \code{a_tau} and \code{c_xi} are adaptive.
Most hyperparameters to be provided are the same as in \code{\link{shrinkTVPVAR}}, however, the following additional ones are required for the dynamic shrinkage prior:
\itemize{
\item \code{iid}:  logical. The default value is \code{TRUE}.
\item \code{a_psi}: numeric vector of length \code{m*p} if const is FALSE and \code{m*p + 1} if const is TRUE. The default value is a vector filled with 0.5.
\item \code{c_psi}: numeric vector of length \code{m*p} if const is FALSE and \code{m*p + 1} if const is TRUE. The default value is a vector filled with 0.5.
\item \code{a_rho}: positive, real number. The default value is 2.
\item \code{b_rho}:positive, real number between 0 and 1. The default value is 0.95.
\item \code{alpha_rho}: numeric vector of length \code{m*p} if const is FALSE and \code{m*p + 1} if const is TRUE. The default value is a vector filled with 0.5.
\item \code{beta_rho}: numeric vector of length \code{m*p} if const is FALSE and \code{m*p + 1} if const is TRUE. The default value is a vector filled with 3.
\item \code{tuning_par_rho}: positive, real number. The default value is 1.
\item \code{adaptive_rho}: logical. If \code{TRUE}, the MH step for rho is adaptive, otherwise it is not. The default value is \code{TRUE}.
\item \code{target_rate_rho}: positive, real number. The default value is 0.44.
\item \code{batch_size_rho}: positive integer. The default value is 50.
\item \code{max_adapt_rho}: positive, real number. The default value is 0.01.
 }}

\item{TVP_params_sigma}{\emph{optional} named list containing hyperparameter values for the TVP prior of the Sigma matrix.
The structure is the same as for \code{TVP_params_beta}. The default values are the same as for \code{TVP_params_beta}.}
}
\value{
A list of class \code{"shrinkDTVPVAR"} containing:
\item{\code{beta}}{an \code{mcmc.tvp.var} object with the VAR coefficient draws.}
\item{\code{beta_mean}}{an \code{mcmc.var} object with the beta_mean draws.}
\item{\code{theta_sr}}{an \code{mcmc.var} object with the theta_sr draws.}
\item{\code{xi2}}{an \code{mcmc.var} object with the xi2 draws.}
\item{\code{c_xi}}{an \code{mcmc} object with the c_xi draws.}
\item{\code{kappa2}}{an \code{mcmc.var} object with the kappa2 draws.}
\item{\code{kappa2_B}}{an \code{mcmc} object with the kappa2_B draws.}
\item{\code{a_xi}}{an \code{mcmc} object with the a_xi draws.}
\item{\code{tau2}}{an \code{mcmc.var} object with the tau2 draws.}
\item{\code{c_tau}}{an \code{mcmc} object with the c_tau draws.}
\item{\code{lambda2}}{an \code{mcmc.var} object with the lambda2 draws.}
\item{\code{lambda2_B}}{an \code{mcmc} object with the lambda2_B draws.}
\item{\code{a_tau}}{an \code{mcmc} object with the a_tau draws.}
\item{\code{Sigma}}{an \code{mcmc.tvp.var} object with the covariance matrix draws.}
\item{\code{psi}}{an \code{mcmc.tvp.var} object with the psi draws.}
\item{\code{rho_p}}{an \code{mcmc.var} object with the rho draws.}
\item{\code{pred_objs}}{a list with objects needed for prediction methods.}
\item{\code{final_lambda}}{an \code{mcmc.var} with the values of lambda at time T of the dynamic shrinkage process. Used for predicting.}
\item{\code{final_lambda_SIGMA}}{an array with the values of lambda of the variance-covariance matrix Sigma at time T of the dynamic shrinkage process.
Used for predicting.}
\item{\code{rho_p_SIGMA}}{an array with the rho_p values of the variance-covariance matrix Sigma. Used for predicting.}
\item{\code{beta_consts}}{a list of \code{mcmc.tvp} objects with the intercept draws (if \code{const} is \code{TRUE}).}
\item{\code{psi_consts}}{a list of \code{mcmc.tvp} objects with the psi draws (if \code{const} is \code{TRUE}).}
\item{\code{data}}{a list with the original data used for estimation.}
}
\description{
\code{shrinkDTVPVAR} samples from the joint posterior distribution of the parameters of a
TVP-VAR-SV model as described in Cadonna et al. (2020) and returns the MCMC draws. The prior on
the VAR coefficients is a dynamic shrinkage prior, as described in Knaus and Frühwirth-Schnatter (2023).
The model can be written as:
\deqn{Y_t = c_t + \Phi_{1,t} Y_{t-1} + \Phi_{2,t} Y_{t-2} + \cdots + \Phi_{p,t} Y_{t-p} + \epsilon_t}
where \eqn{\epsilon_t \sim \mathcal{N}_m(0, \Sigma_t)}.
}
\details{
The elements of the VAR coefficients \eqn{\Phi_{i,t}} are assumed to follow component-wise random walks.

For further details concerning the algorithms and the model please refer to the papers by Cadonna et al. (2020) and
Knaus and Frühwirth-Schnatter (2023).
}
\examples{
\dontrun{
set.seed(123)
sim <- simTVPVAR(p = 2)
data <- sim$data

res <- shrinkDTVPVAR(data, p = 2)

# Visualize the results
plot(res)
plot(res$theta_sr)

# Change prior to triple gamma
res2 <- shrinkDTVPVAR(data, p = 2, mod_type = "triple")

# Modify the hyperparameter setup
hyperparam <- list(learn_a_xi = FALSE, learn_c_xi = FALSE,
                  learn_a_tau = FALSE, learn_c_tau = FALSE,
                  a_xi = 0.5, c_xi = 0.5, a_tau = 0.5, c_tau = 0.5)

res3 <- shrinkDTVPVAR(data, p = 2, mod_type = "triple",
                    TVP_params_beta = hyperparam,
                    TVP_params_sigma = hyperparam)
}

}
\references{
Cadonna, A., Frühwirth-Schnatter, S., & Knaus, P. (2020). Triple the Gamma—A Unifying Shrinkage Prior for Variance and Variable Selection in Sparse State Space and TVP Models. \emph{Econometrics}, 8(2), 20.

Knaus, P., Bitto-Nemling, A., Cadonna, A., & Frühwirth-Schnatter, S. (2021). Shrinkage in the Time-Varying Parameter Model Framework Using the \code{R} Package \code{shrinkTVP}. \emph{Journal of Statistical Software}, 100(13), 1–32.

Knaus, P., & Frühwirth-Schnatter, S. (2023). The Dynamic Triple Gamma Prior as a Shrinkage Process Prior for Time-Varying Parameter Models. \emph{arXiv preprint} arXiv:2312.10487.
}
\seealso{
\code{\link{TV_heatmap}}, \code{\link{density_plotter}}, \code{\link{state_plotter}}
}
\author{
Peter Knaus \email{peter.knaus@wu.ac.at}
}
