% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simplex_tree.R
\name{plot.simplextree}
\alias{plot.simplextree}
\alias{plot.Rcpp_SimplexTree}
\title{Plots the simplex tree}
\usage{
\method{plot}{Rcpp_SimplexTree}(
  x,
  coords = NULL,
  vertex_opt = NULL,
  text_opt = NULL,
  edge_opt = NULL,
  polygon_opt = NULL,
  color_pal = NULL,
  maximal = TRUE,
  by_dim = TRUE,
  add = FALSE,
  ...
)
}
\arguments{
\item{x}{a simplex tree.}

\item{coords}{Optional (n x 2) matrix of coordinates, where n is the number of 0-simplices.}

\item{vertex_opt}{Optional parameters to modify default vertex plotting options. Passed to \code{\link[graphics]{points}}.}

\item{text_opt}{Optional parameters to modify default vertex text plotting options. Passed to \code{\link[graphics]{text}}.}

\item{edge_opt}{Optional parameters to modify default edge plotting options. Passed to \code{\link[graphics]{segments}}.}

\item{polygon_opt}{Optional parameters to modify default k-simplex plotting options for k > 1. Passed to \code{\link[graphics]{polygon}}.}

\item{color_pal}{Optional vector of colors. See details.}

\item{maximal}{Whether to draw only the maximal faces of the complex. Defaults to true.}

\item{by_dim}{Whether to apply (and recycle or truncate) the color palette to the dimensions rather than to the individual simplices. Defaults to true.}

\item{add}{Whether to add to the plot or redraw. Defaults to false. See details.}

\item{...}{unused}
}
\description{
Plots the simplex tree
}
\details{
This function allows generic plotting of simplicial complexes using base \code{\link[graphics:graphics-package]{graphics}}.\cr
If not (x,y) coordinates are supplied via \code{coords}, a default layout is generated via phyllotaxis arrangement. This layout is 
not in general does not optimize the embedding towards any usual visualization criteria e.g. it doesn't try to separate connected components, 
minimize the number of crossings, etc. For those, the user is recommended to look in existing code graph drawing libraries, e.g. igraphs 'layout.auto' function, etc. 
The primary benefit of the default phyllotaxis arrangement is that it is deterministic and fast to generate. 
\cr
All parameters passed via list to \code{vertex_opt}, \code{text_opt}, \code{edge_opt}, \code{polygon_opt} 
override default parameters and are passed to \code{\link[graphics]{points}}, \code{\link[graphics]{text}}, \code{\link[graphics]{segments}}, 
and \code{\link[graphics]{polygon}}, respectively.\cr
\cr
If \code{add} is true, the plot is not redrawn. \cr
\cr
If \code{maximal} is true, only the maximal simplices are drawn. \cr
\cr
The \code{color_pal} argument controls how the simplicial complex is colored. It can be specified in multiple ways.
\enumerate{
  \item A vector of colors of length \emph{dim+1}, where \emph{dim}=\code{x$dimension}
  \item A vector of colors of length \emph{n}, where \emph{n}=\code{sum(x$n_simplices)}
  \item A named list of colors
}
Option (1) assigns every simplex a color based on its dimension. \cr
\cr
Option (2) assigns each individual simplex a color. The vector must be specified in level-order 
(see \code{\link{ltraverse}} or examples below). \cr
\cr
Option (3) allows specifying individual simplices to draw. It expects a named list, where the names
must correspond to simplices in \code{x} as comma-separated strings and whose values are colors. If 
option (3) is specified, this method will \emph{only} draw the simplices given in \code{color_pal}.\cr
\cr
If \code{length(color_pal)} does not match the dimension or the number of simplices in the complex, 
the color palette is recyled and simplices are as such.
}
\examples{

## Simple 3-simplex 
st <- simplex_tree() \%>\% insert(list(1:4))

## Default is categorical colors w/ diminishing opacity
plot(st)

## If supplied colors have alpha defined, use that 
vpal <- rainbow(st$dimension + 1)
plot(st, color_pal = vpal)

## If alpha not supplied, decreasing opacity applied
plot(st, color_pal = substring(vpal, first=1, last=7))

## Bigger example; observe only maximal faces (+vertices and edges) are drawn
st <- simplex_tree(list(1:3, 2:5, 5:9, 7:8, 10))
plot(st, color_pal = rainbow(st$dimension + 1))

## If maximal == FALSE, every simplex is drawn (even on top of each other)
vpal <- rainbow(st$dimension + 1)[c(1,2,5,4,3)]
pal_alpha <- c(1, 1, 0.2, 0.35, 0.35)
vpal <- sapply(seq_along(vpal), function(i) adjustcolor(vpal[i], alpha.f = pal_alpha[i]))
plot(st, color_pal = vpal, maximal = FALSE)

## You can also color each simplex individually by supplying a vector 
## of the same length as the number of simplices. 
plot(st, color_pal = sample(rainbow(sum(st$n_simplices))))

## The order is assumed to follow the level order traversal (first 0-simplices, 1-, etc.)
## This example colors simplices on a rainbow gradient based on the sum of their labels
si_sum <- straverse(st \%>\% level_order, sum) 
rbw_pal <- rev(rainbow(50, start=0,end=4/6))
plot(st, color_pal=rbw_pal[cut(si_sum, breaks=50, labels = FALSE)])

## This also makes highlighting simplicial operations fairly trivial 
four_cofaces <- as.list(cofaces(st, 4))
coface_pal <- straverse(level_order(st), function(simplex){ 
    ifelse(list(simplex) \%in\% four_cofaces, "orange", "blue") 
})
plot(st, color_pal=unlist(coface_pal))

## You can also give a named list to draw individual simplices. 
## **Only the maximal simplices in the list are drawn** 
blue_vertices <- structure(as.list(rep("blue", 5)), names=as.character(seq(5, 9)))
plot(st, color_pal=append(blue_vertices, list("5,6,7,8,9"="red")))
}
