% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.R
\name{fit_spm}
\alias{fit_spm}
\alias{fit_spm.spm}
\alias{fit_spm2}
\title{Fitting an underlying continuous process to areal data}
\usage{
fit_spm(x, ...)

\method{fit_spm}{spm}(
  x,
  model,
  theta_st,
  nu = NULL,
  kappa = 1,
  mu2 = 1.5,
  apply_exp = FALSE,
  opt_method = "Nelder-Mead",
  control_opt = list(),
  comp_hess = TRUE,
  ...
)

fit_spm2(
  x,
  model,
  nu,
  kappa = 1,
  mu2 = 1.5,
  comp_hess = TRUE,
  phi_min,
  phi_max,
  nphi = 10,
  cores = getOption("mc.cores", 1L)
)
}
\arguments{
\item{x}{An object of type \code{spm}. The dimension of \code{theta_st}
depends on the number of variables analyzed and whether the input is an
\code{spm} object.}

\item{...}{Additional parameters passed to \link[stats:optim]{stats::optim}.}

\item{model}{A \code{character} scalar indicating the covariance function
family. Options are \code{c("matern", "pexp", "gaussian", "spherical",
  "gw")}.}

\item{theta_st}{A named \code{numeric} vector containing initial parameter
values.}

\item{nu}{A \code{numeric} value specifying the \eqn{\nu} parameter for the
Matern or Powered Exponential covariance functions. If \code{model} is
"matern" and \code{nu} is not provided, it defaults to 0.5. If
\code{model} is "pexp" and \code{nu} is not provided, it defaults to 1. In
both cases, this results in the exponential covariance function.}

\item{kappa}{\eqn{\kappa \in \{0, \ldots, 3 \}} parameter for the GW
covariance function.}

\item{mu2}{The smoothness parameter \eqn{\mu} for the GW function.}

\item{apply_exp}{A \code{logical} scalar indicating whether to exponentiate
non-negative parameters.}

\item{opt_method}{A \code{character} scalar specifying the optimization
algorithm (see \link[stats:optim]{stats::optim}).}

\item{control_opt}{A named \code{list} of control arguments for the
optimization algorithm (see \link[stats:optim]{stats::optim}).}

\item{comp_hess}{A \code{logical} indicating whether to compute the Hessian
matrix.}

\item{phi_min}{A \code{numeric} scalar representing the minimum \eqn{phi}
value for grid search.}

\item{phi_max}{A \code{numeric} scalar representing the maximum \eqn{phi}
value for grid search.}

\item{nphi}{A \code{numeric} scalar indicating the number of \eqn{phi}
values for grid search.}

\item{cores}{An \code{integer} scalar indicating the number of cores to use.
Defaults to \code{getOption("mc.cores")}. No effect on Windows.}
}
\value{
An \code{spm_fit} object containing the estimated model parameters.
}
\description{
This function fits a spatial process model to areal data by
estimating the parameters of an underlying continuous process. It
leverages the \link[stats:optim]{stats::optim} function for Maximum Likelihood estimation,
providing flexibility in optimization algorithms and control parameters.
}
\details{
The function utilizes optimization algorithms from \link[stats:optim]{stats::optim}
to determine Maximum Likelihood estimators (MLEs) and their standard
errors for a model adapted for areal data. Users can customize the
optimization process by providing control parameters through the
\code{control_opt} argument (a named list, see \link[stats:optim]{stats::optim} for details),
specifying lower and upper bounds for parameters, and selecting the
desired optimization algorithm via \code{opt_method} (must be available in
\link[stats:optim]{stats::optim}).

Additionally, the function supports various covariance functions,
including Matern, Exponential, Powered Exponential, Gaussian, and
Spherical. The \code{apply_exp} argument, a logical scalar, allows for
exponentiation of non-negative parameters, enabling optimization over the
entire real line.

The underlying model assumes a point-level process:
\deqn{Y(\mathbf{s}) = \mu + S(\mathbf{s})} where
\eqn{S ~ GP(0, \sigma^2 C(\lVert \mathbf{s} - \mathbf{s}_2 \rVert; \theta))}.
The observed areal data is then assumed to behave as the average of the process
over each area:
\deqn{Y(B) = \lvert B \rvert^{-1} \int_{B} Y(\mathbf{s}) \, \textrm{d} \mathbf{s}}.
}
\examples{
data(liv_lsoa) ## loading the LSOA data

msoa_spm <- sf_to_spm(sf_obj = liv_msoa, n_pts = 500,
                     type = "regular", by_polygon = FALSE,
                     poly_ids = "msoa11cd",
                     var_ids = "leb_est")
## fitting model
theta_st_msoa <- c("phi" = 1) # initial value for the range parameter

fit_msoa <-
  fit_spm(x = msoa_spm,
          theta_st = theta_st_msoa,
          model = "matern",
          nu = .5,
          apply_exp  = TRUE,
          opt_method = "L-BFGS-B",
          control    = list(maxit = 500))

AIC(fit_msoa)

summary_spm_fit(fit_msoa, sig = .05)

}
