% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SocialRanking.R
\name{doRanking}
\alias{doRanking}
\title{Create a \code{SocialRanking} object}
\usage{
doRanking(scores, compare = NULL, decreasing = TRUE)
}
\arguments{
\item{scores}{A vector or list representing each element's score. If \code{names(scores)} is not \code{NULL}, those will be used as element names.
Else a number sequence corresponding to the elements is generated.}

\item{compare}{Optional comparison function taking in two elements and returning a numerical value based on the relation between
these two elements. If set to \code{NULL}, the default \code{\link[=order]{order()}} function is called. See details for more information.}

\item{decreasing}{If \code{TRUE} (default), elements with higher scores are ranked higher.}
}
\value{
A list of type \code{SocialRanking}.
Each element of the list contains a \link[base:c]{vector} of elements in \code{powerRelation$elements} that are indifferent to one another.
}
\description{
Rank elements based on their scores.
}
\details{
All ranking solutions in the package are tied to the scores or score vectors of the elements.
For these kinds of solutions, \code{doRanking()} offers a simple way that turns a (named) vector or list of scores for each element into a \code{SocialRanking} object.
For example, \code{doRanking(c(a=1,b=2))} produces \code{b > a} (\eqn{b P^\succsim a}{bPa}), because \code{b} with a score of \code{2} should be placed higher than \code{a} with a score of \code{1}.

Ranking solutions in the package include \code{\link[=lexcelRanking]{lexcelRanking()}}, \code{\link[=ordinalBanzhafRanking]{ordinalBanzhafRanking()}} and \code{\link[=L1Ranking]{L1Ranking()}}, among others.
These functions take a power relation, calculate the scores of each element and returns a \code{SocialRanking} object.

R natively supports sorting for \link[base:c]{vectors}, but not for \link[base:list]{lists}.
If the use of lists is necessary, or if the native sort method in vectors does not produce the desired results, there are two possible ways to solve this:
\enumerate{
\item by the introduction of custom S3 classes, or
\item by setting the \code{compare} parameter in \code{doRanking()}.
}

For S3 classes, the class for the score object has to be set and the \code{==} and \code{>} (and \code{[} for lists) operators overloaded.
I.e., \code{\link[=lexcelScores]{lexcelScores()}} returns a list with the custom class \code{LexcelScores} that implements \verb{==.LexcelScores}, \verb{>.LexcelScores}, \verb{[.LexcelScores} and \code{is.na.LexcelScores}.

In cases where we only want to experiment, introducing new S3 classes can be cumbersome.
As an alternative, the \code{compare} parameter can be assigned a function.
This function must take two parameters, i.e., \verb{function(a, b)}, where \code{a} and \code{b} are the scores of two arbitrary elements.
The function then must return one of the following:
\itemize{
\item \verb{> 0} (positive value) if score \code{a} is ranked higher than score \code{b},
\item \verb{< 0} (negative value) if score \code{a} is ranked lower than score \code{b}, or
\item \verb{= 0} if both scores \code{a} and \code{b} are considered equal.
}

In \code{doRanking(c(a=3,b=2,c=2), compare = function(a,b) a - b)}, the compare function returns a positive value of the first parameter is larger than the second.
\code{a} has the highest value and will there for be ranked highest, \code{a > b ~ c}.

Conversely, \code{doRanking(c(a=3,b=2,c=2), compare = function(a,b) b - a)} favors elements with lower scores, resulting in the element ranking \code{b ~ c > a}.
}
\examples{
doRanking(c(a=1,b=2))
# b > a

doRanking(c(a=2,b=2))
# a ~ b

# a custom ranking function. Here, we implement the following ranking solution:
# disregard any big coalitions and only rank elements based on their individual performances
# iRj if and only if {i} >= {j}
singletonRanking <- function(pr) {
  scores <- sapply(pr$elements, equivalenceClassIndex, powerRelation = pr)
  # note that coalitions in higher indexed equivalence classes are less preferable
  # hence, scores should be sorted in an increasing order
  doRanking(scores, decreasing = FALSE)
}

pr <- as.PowerRelation("abc > ab > ac > b ~ c ~ bc > a")
singletonRanking(pr)
# b ~ c > a

# a reverse lexcel ranking, where vectors are compared right to left
# here, we introduce a compare function. It returns:
# * 0, if a and b are identical
# * a positive value, if a[i] > b[i] and every value after that is equal
# * a negative value, if a[i] < b[i] and every value after that is equal
reverseLexcelCompare <- function(a, b) {
  i <- which(a != b) |> rev()
  if(length(i) == 0) 0
  else a[i[1]] - b[i[1]]
}

scores <- unclass(cumulativeScores(pr))

# R cannot natively sort a class. Instead:
# Method 1 - utilize the compare parameter
doRanking(scores, compare = reverseLexcelCompare)


# Method 2 - introduce S3 class
`[.RevLex` <- function(x, i, ...) structure(unclass(x)[i], class = "RevLex")
`==.RevLex` <- function(a, b) reverseLexcelCompare(a[[1]],b[[1]]) == 0
`>.RevLex` <- function(a, b) reverseLexcelCompare(a[[1]],b[[1]]) > 0
is.na.RevLex <- function(x) FALSE
doRanking(structure(scores, class = "RevLex"))

stopifnot(
  doRanking(scores, compare = reverseLexcelCompare) ==
  doRanking(structure(scores, class = "RevLex"))
)

}
\seealso{
\code{\link[=SocialRanking]{SocialRanking()}}
}
