#' Options for the squids package
#'
#' The `squids::opts` object contains three functions to set, get, and reset
#' options used by the zirconia package. Use `squids::opts$set` to set options,
#' `squids::opts$get` to get options, or `squids::opts$reset` to reset specific or
#' all options to their default values.
#'
#' It is normally not necessary to get or set `squids` options.
#'
#' The following arguments can be passed:
#'
#' \describe{
#'   \item{...}{For `squids::opts$set`, the dots can be used to specify the options
#'   to set, in the format `option = value`, for example, `utteranceMarker = "\n"`. For
#'   `squids::opts$reset`, a list of options to be reset can be passed.}
#'   \item{option}{For `squids::opts$set`, the name of the option to set.}
#'   \item{default}{For `squids::opts$get`, the default value to return if the
#'   option has not been manually specified.}
#' }
#'
#' The following options can be set:
#'
#' \describe{
#'
#' \item{silent}{Whether to be silent or chatty.}
#' \item{encoding}{The default encoding when reading or writing files.}
#' \item{preventOverwriting}{Whether to be prevent overwriting of existing files.}
#' \item{debug}{Sometimes used to display debugging information.}
#'
#' }
#'
#' @aliases opts set get reset
#'
#' @usage opts
#'
#' @examples ### Get the default 'silent' setting
#' squids::opts$get(silent);
#'
#' ### Set it to FALSE (to be chatty)
#' squids::opts$set(silent = FALSE);
#'
#' ### Check that it worked
#' squids::opts$get(silent);
#'
#' ### Reset this option to its default value
#' squids::opts$reset(silent);
#'
#' ### Check that the reset worked, too
#' squids::opts$get(silent);
#'
#' @export
opts <- list();

opts$set <- function(...) {
  dots <- list(...);
  dotNames <- names(dots);
  names(dots) <-
    paste0("squids.", dotNames);
  if (all(dotNames %in% names(opts$defaults))) {
    do.call(options,
            dots);
  } else {
    stop("Option '", option, "' is not a valid (i.e. existing) option for squids!");
  }
}

opts$get <- function(option, default=FALSE) {
  option <- as.character(substitute(option));
  if (!option %in% names(opts$defaults)) {
    stop("Option '", option, "' is not a valid (i.e. existing) option for squids!");
  } else {
    return(getOption(paste0("squids.", option),
                     opts$defaults[[option]]));
  }
}

opts$reset <- function(...) {
  optionNames <-
    unlist(lapply(as.list(substitute(...)),
                  as.character));
  if (length(optionNames) == 0) {
    do.call(opts$set,
            opts$defaults);
  } else {
    prefixedOptionNames <-
      paste0("squids.", optionNames);
    if (all(optionNames %in% names(opts$defaults))) {
      do.call(opts$set,
              opts$defaults[optionNames]);
    } else {
      invalidOptions <-
        !(optionNames %in% names(opts$defaults));
      stop("Option(s) ", vecTxtQ(optionNames[invalidOptions]),
           "' is/are not a valid (i.e. existing) option for squids!");
    }
  }
}

opts$defaults <-
  list(

    ### Used throughout for working with files
    encoding = "UTF-8",
    preventOverwriting = FALSE,

    ### Used throughout for debugging mode
    debug = FALSE,

    ### Used throughout for suppressing messages
    silent = TRUE

  );
