% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bootstrap.R
\name{bootstrap}
\alias{bootstrap}
\alias{bootstrap_msel}
\alias{bootstrap_combine_msel}
\title{Bootstrap for msel function}
\usage{
bootstrap_msel(
  object,
  iter = 100,
  opt_type = "optim",
  opt_args = NULL,
  is_ind = FALSE,
  n_sim = 1000,
  n_cores = 1
)

bootstrap_combine_msel(...)
}
\arguments{
\item{object}{an object of class 'msel'.}

\item{iter}{the number of bootstrap iterations.}

\item{opt_type}{the same as \code{opt_type} argument of the
\code{\link[switchSelection]{msel}} function.}

\item{opt_args}{the same as \code{opt_args} argument of the
\code{\link[switchSelection]{msel}} functions.}

\item{is_ind}{logical; if \code{TRUE} then the function also returns a
numeric matrix of indexes of observations used in the bootstrap samples.}

\item{n_sim}{the same as \code{n_sim} argument of the
\code{\link[switchSelection]{msel}} function.}

\item{n_cores}{the same as \code{n_cores} argument of the 
\code{\link[switchSelection]{msel}} function.}

\item{...}{objects returned by function 
\code{\link[switchSelection]{bootstrap_msel}} to be combined into a
single object.}
}
\value{
Function \code{\link[switchSelection]{bootstrap_msel}}
returns an object of class \code{"bootstrap_msel"}.
This object is a list which may contain the following elements:
\itemize{
\item \code{par} - a numeric matrix such that \code{par[b, ]} is a vector 
of the estimates of the parameters of the model estimated via
\code{\link[switchSelection]{msel}} function on the \code{b}-th bootstrap
sample.
\item \code{iter} - the number of the bootstrap iterations.
\item \code{cov} - bootstrap estimate of the covariance matrix which 
equals to \code{cov(par)}.
\item \code{ind} - a numeric matrix such that \code{ind[, b]} stores the 
indexes of the observations from \code{object$data} included into the 
\code{b}-th bootstrap sample.
}
Function \code{\link[switchSelection]{bootstrap_combine_msel}} returns the 
object which combines several objects returned by the
\code{\link[switchSelection]{bootstrap_msel}} function into a single
object.
}
\description{
Function \code{\link[switchSelection]{bootstrap_msel}} 
provides bootstrap estimates of the parameters of the model estimated via
the \code{\link[switchSelection]{msel}} function.
Function \code{\link[switchSelection]{bootstrap_combine_msel}} allows to
combine several objects of class \code{'bootstrap_msel'}.
}
\details{
The function generates \code{iter} bootstrap samples and 
estimates the parameters \eqn{\theta} of the model by using each of
these samples. Estimate \eqn{\hat{\theta}^{(b)}} from the \eqn{b}-th of 
these samples is stored as the \code{b}-th row of the numeric matrix
\code{par} which is an element of the returned object.

Use \code{\link[switchSelection]{update_msel}} function to
transfer the bootstrap estimates to the object of class \code{'msel'}.
}
\examples{
# -------------------------------
# Bootstrap for the probit model
# -------------------------------
\donttest{
# ---
# Step 1
# Simulation of data
# ---

# Load required package
library("mnorm")

# Set seed for reproducibility
set.seed(123)

# The number of observations
n <- 100

# Regressors (covariates)
w1 <- runif(n = n, min = -1, max = 1)
w2 <- runif(n = n, min = -1, max = 1)

# Random errors
u <- rnorm(n = n, mean = 0, sd = 1)

# Coefficients
gamma <- c(-1, 2)

# Linear index
li <- gamma[1] * w1 + gamma[2] * w2

# Latent variable
z_star <- li + u

# Cuts
cuts <- c(-1, 0.5, 2)

# Observable ordered outcome
z                                           <- rep(0, n)
z[(z_star > cuts[1]) & (z_star <= cuts[2])] <- 1
z[(z_star > cuts[2]) & (z_star <= cuts[3])] <- 2
z[z_star > cuts[3]]                         <- 3
table(z)

# Data
data <- data.frame(w1 = w1, w2 = w2, z = z)

# ---
# Step 2
# Estimation of the parameters
# ---

# Estimation
model <- msel(formula = z ~ w1 + w2, data = data)
summary(model)
 
# ---
# Step 3
# Bootstrap
# ---

# Perform bootstrap
bootstrap <- bootstrap_msel(model)

# Test the hypothesis that H0: gamma[2] = -2gamma[1]
# by using the t-test and with bootstrap p-values
fn_test <- function(object)
{
  gamma <- coef(object, eq = 1)
  return(gamma[2] + 2 * gamma[1])
}
b <- test_msel(object    = model, 
               fn        = fn_test,
               test      = "t", 
               method    = "bootstrap",
               ci        = "percentile",
               se_type   = "bootstrap",
               bootstrap = bootstrap)
summary(b)

# Replicate the analysis with the additional 20 bootstrap iterations
bootstrap2    <- bootstrap_msel(model, iter = 20)
bootstrap_new <- bootstrap_combine_msel(bootstrap, bootstrap2)
b2 <- test_msel(object    = model, 
                fn        = fn_test,
                test      = "t", 
                method    = "bootstrap",
                ci        = "percentile",
                se_type   = "bootstrap",
                bootstrap = bootstrap)
summary(b2)
}
 
}
