#' Calculate Time-dependent ROC and AUC with competing risk
#'
#' This function calculates the time-dependent sensitivity and specificity and area under the curve (AUC)
#'        using precalculated weights by \code{td.kw.cr}.
#'
#' @param X a numeric vector of risk score ranging from 0 to 1 in the same length as \code{Y} and \code{delta}. A higher value indicates higher
#'  higher risk of the event. It can be a biomarker value, a function of multiple value, or the predicted cumulative incidence
#'  probability from a copeting risk regression model that we want to evaluate.
#' @param W.prim a numeric vector of weight for the primary event you want to study. It has the same length as \code{X}.
#'        It should be generated by \code{td.kw.cr} with \code{event.code = 1}.
#' @param W.cmp a numeric vector of weight for the competing event. It has the same length as \code{X}.
#'        It should be generated by \code{td.kw.cr} with \code{event.code = 2}.
#' @param n.grid a positive integer, the number of grid points used when calculating the ROC curve. The default is \code{1000}.
#' @param cut.off a vector of risk score cut-off values at which sensitivity and specificity will be calculated. Default is \code{NULL},
#'        and we set cut off point by  \code{n.grid} to calculate sensitivity and specificity.
#' @param method
#'
#' @details This function read in the risk score value \code{X}, estimated conditional probability for primary event \code{W.prim},
#'        and estimated conditional probability for competing event \code{W.cmp} to calculate sensitivity and specificity
#'        for a series specified grid points. Based on the definition of controls mentioned in Wu and Li, 2018, we separately
#'        calculate specificity and corresponding AUC for each definition. In addition, this function returns both the
#'        AUC estimated by trapezoidal integration and AUC estimated by nonparametric framework mentioned in Wu and Li, 2018.
#'
#' @seealso \code{\link[survival]{survfit}}
#'
#' @return Returns a list of the following items:
#
#' @return a list of \code{AUC.A.integral} estimated by trapezoidal integration for definition A,
#'        \code{AUC.A.empirical} estimated by nonparametric framework for definition A (Wu and Li, 2018),
#'         \code{AUC.B.integral} estimated by trapezoidal integration for definition B,
#'        \code{AUC.B.empirical} estimated by nonparametric framework for definition B (Wu and Li, 2018),
#'        and a data frame \code{ROC} with dimension \code{(2+n.grid) x 4} with columns \code{cut.off}, \code{sens}, \code{specA} and \code{specB}.
#' @importFrom survival survfit
#' @keywords internal
#'

AUC.cr <- function(X, W.prim, W.cmp,
                   cut.off = NULL, n.grid = 1000, method) {
  ## W.prim is the weight for primary event
  ## W.cmp is the weight for competing event
  # Default use sample marker values as cut.offs
  n.grid <- min(n.grid, length(X))
  if (is.null(cut.off)) {
    cut.off <- c(-Inf, seq(min(X), max(X), length.out = n.grid), Inf)
  }

  sens <- specA <- specB <- NULL

  negA <- 1 - W.prim
  negB <- 1 - W.prim - W.cmp
  sens_denom <- sum(W.prim)
  specA_denom <- sum(negA)
  specB_denom <- sum(negB)

  X_ge_cutoff <- outer(X, cut.off, ">")
  X_le_cutoff <- outer(X, cut.off, "<=")

  sens <- colSums(X_ge_cutoff * W.prim) / sens_denom
  specA <- colSums(X_le_cutoff * negA) / specA_denom
  specB <- colSums(X_le_cutoff * negB) / specB_denom

  ROC <- data.frame(
    cut.off = cut.off,
    sens = sens,
    specA = specA,
    specB = specB
  )

  AUC.A.integral <- AUC.B.integral <- AUC.A.empirical <- AUC.B.empirical <- NA

  if(method != "integral"){

    AUC2 <- AUC_calc2(X, W.prim, W.cmp)
    AUC.A.empirical <- AUC2[1]
    AUC.B.empirical <- AUC2[2]
  }

  if(method != "empirical"){

    AUC.A.integral <- AUC_calc_integral(sens, specA)
    AUC.B.integral <- AUC_calc_integral(sens, specB)
  }

  res <- list(AUC.A.integral = AUC.A.integral,
              AUC.B.integral = AUC.B.integral,
              AUC.A.empirical = AUC.A.empirical,
              AUC.B.empirical = AUC.B.empirical,
              ROC = ROC)

  return(res)
}
