#' Generate a Summary Report
#'
#' Generates a summary report containing the output of autoExtract(), normalityPlots() and comparisonPlots().
#'
#' @param audioData A data.frame generated by autoExtract() function.
#' @param savePath Character string indicating the full path to the folder to which we want to save the generated report. By default it is set to the current working directory.
#' @param includeDimensions Logical value indicating whether Dimensions should be also included as a factor or not. Default corresponds to FALSE.
#' @param avoidNormalCheck Logical vector, indicating if and what variables' distribution were transformed to normal. By default it is set to FALSE for each of the measures. Alternatively, you can set it to TRUE to avoid checking normality for all the measures.
#' @param filename Optional character string indicating the file name of the generated report. Default corresponds to "voiceR_report.html".
#' @return html report file, which is saved in the selected path, but returns nothing.
#' @examples
#' \donttest{
#' autoReport(testAudioData = testAudioData)
#' }
#'
#' @importFrom rmarkdown render html_document
#' @export


autoReport <- function(audioData, savePath = getwd(), includeDimensions = FALSE, avoidNormalCheck  = c(), filename = "voiceR_report.html"){

  if(!is.character(savePath) || !file.exists(savePath)) stop("Invalid savePath!")
  if(!is.data.frame(audioData)) stop("audioData must be a data.frame")
  if(!is.character(filename)) stop("filename must be a string!")
  if(!is.logical(includeDimensions)) stop("includeDimensions must be a boolean!")




  #Use the location for the report template (Rmd file)
  src <- system.file("AppShiny/report.Rmd", package = "voiceR")

  measures <- c("duration", "voice_breaks_percent", "RMS_env", "mean_loudness", "mean_F0", "sd_F0", "mean_entropy", "mean_HNR")
  if(length(avoidNormalCheck) == 1 && avoidNormalCheck) avoidNormalCheck <- rep(TRUE, length(measures))

  # temporarily switch to the temp dir, in case you do not have write
  # permission to the current working directory
  owd <- setwd(tempdir())
  on.exit(setwd(owd))
  #copy file
  file.copy(src, 'report.Rmd', overwrite = TRUE)
  #Create comparison and normality plots
  comparisons <- comparisonPlots(audioData, by = "Condition")
  normalPlots <- normalityPlots(audioData)
  comparisons2 <- list()
  normalPlots2 <- list()

  for (i in 1:length(measures)) {
    measure <- measures[i]
    comparisons2[[i]] <- comparisons[[measure]]
    normalPlots2[[i]] <- normalPlots[[measure]]

  }

  if(length(avoidNormalCheck) == 0)
    avoidNormalCheck <- rep(FALSE, 8)
  #Send these parameters to the R markdown document and render the file
  params <- list(audioData = audioData, comparisons = comparisons2, normalPlots = normalPlots2, includeDimensions = includeDimensions, avoidNormalCheck = avoidNormalCheck)
  out <- render('report.Rmd', html_document(), params = params, envir = new.env())
  file.rename(out, paste0(savePath, filename))
}

