#' Choose properties to return for pages from the action API
#'
#' See [API:Properties](https://www.mediawiki.org/wiki/API:Properties) for a
#' list of available properties. Many have additional parameters to control
#' their behavior, which can be passed to this function as named arguments.
#'
#' [query_page_properties] is not useful on its own. It must be combined with a
#' [query_by_] function or [query_generate_pages] to specify which pages
#' properties are to be returned. It should be noted that many of the
#' [API:Properties](https://www.mediawiki.org/wiki/API:Properties) modules can
#' themselves be used as generators. If you wish to use a property module in
#' this way, then you must use [query_generate_pages], passing the name of the
#' property module as the `genenerator`.
#'
#' @param .req A httr2_request, e.g. generated by `wiki_action_request`
#' @param property The property to request
#' @param ... <[`dynamic-dots`][rlang::dyn-dots]> Additional parameters to pass, e.g. to modify what is returned by
#'   the property request
#'
#' @return An HTTP response: an S3 list with class httr2_request
#' @export
#'
#' @seealso [gracefully()]
#'
#' @examples
#' # Search for articles about seagulls and retrieve their number of
#' # watchers
#'
#' resp <- wiki_action_request() %>%
#'   query_generate_pages("search", gsrsearch = "seagull") %>%
#'   query_page_properties("info", inprop = "watchers") %>%
#'   gracefully(next_batch) %>%
#'   dplyr::select(pageid, ns, title, watchers)
#' resp
query_page_properties <- function(.req, property, ...) {
  check_module(property, "prop")
  query <- set_action(.req, "prop", property, ...)
}

#' @rdname query_page_properties
#' @export
list_all_property_modules <- function() {
  schema_query_modules %>%
    dplyr::filter(group == "prop") %>%
    dplyr::select(name)
}
