## Domain Registry Interface, Auxiliary Contacts Extension Mapping for EPP
##
## Copyright (c) 2016,2018 Patrick Mevzek <netdri@dotandco.com>. All rights reserved.
##
## This file is part of Net::DRI
##
## Net::DRI is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## See the LICENSE file that comes with this distribution for more details.
####################################################################################################

package Net::DRI::Protocol::EPP::Extensions::CentralNic::AuxContact;

use strict;
use warnings;
use feature 'state';

use Net::DRI::Util;

####################################################################################################

sub register_commands
{
 my ($class,$version)=@_;
 state $rcmds = { 'domain' => {
                                'info'   => [ undef,    \&info_parse ],
                                'create' => [ \&create, undef ],
                                'update' => [ \&update, undef ],
                              },
                };
 return $rcmds;
}

sub setup
{
 my ($class,$po,$version)=@_;
 state $rns = { 'auxcontact' => 'urn:ietf:params:xml:ns:auxcontact-0.1' };
 $po->ns($rns);
 return;
}

sub implements { return 'https://gitlab.centralnic.com/centralnic/epp-auxcontact-extension/blob/1b7a0e935523b306387d236908017ea7ba72f794/draft-brown-auxcontact.txt'; }

####################################################################################################

sub info_parse
{
 my ($po,$otype,$oaction,$oname,$rinfo)=@_;
 my $mes=$po->message();
 return unless $mes->is_success();

 my $ns=$mes->ns('auxcontact');
 my $data=$mes->get_extension('auxcontact', 'infData');
 return unless defined $data;

 my $cs=$rinfo->{domain}->{$oname}->{contact};
 my %ccache=map { $_->id() => $_ } $cs->get_all();
 foreach my $el (Net::DRI::Util::xml_list_children($data))
 {
  my ($name,$node)=@$el;
  if ($name eq 'contact')
  {
   my $id=$node->textContent();
   $ccache{$id}=$po->create_local_object('contact')->srid($id) unless exists $ccache{$id};
   $cs->add($ccache{$id},$node->getAttribute('type'));
  }
 }

 return;
}

sub _build_contacts
{
 my ($epp, $cs) = @_;

 my @d;
 my %r=map { $_ => 1 } ('registrant', $epp->core_contact_types());
 foreach my $t (sort { $a cmp $b } grep { ! exists $r{$_} } $cs->types())
 {
  my @o=$cs->get($t);
  push @d,map { [ 'auxcontact:contact',$_->srid(),{'type'=>$t}] } @o;
 }

 return @d;
}

sub create
{
 my ($epp,$domain,$rd)=@_;

 return unless Net::DRI::Util::has_contact($rd);
 my $cs=$rd->{contact};

 my @d = _build_contacts($epp, $cs);
 return unless @d;

 my $mes=$epp->message();
 my $eid=$mes->command_extension_register('auxcontact', 'create');
 $mes->command_extension($eid, \@d);
 return;
}

sub update
{
 my ($epp,$domain,$todo)=@_;

 my $cadd=$todo->add('contact');
 my $cdel=$todo->del('contact');
 my (@add,@del);
 push @add,_build_contacts($epp,$cadd) if Net::DRI::Util::isa_contactset($cadd);
 push @del,_build_contacts($epp,$cdel) if Net::DRI::Util::isa_contactset($cdel);

 return unless @add || @del;

 my @d;
 push @d,['auxcontact:rem',@del] if @del;
 push @d,['auxcontact:add',@add] if @add;

 my $mes=$epp->message();
 my $eid=$mes->command_extension_register('auxcontact', 'update');
 $mes->command_extension($eid, \@d);
 return;
}

####################################################################################################
1;

__END__

=pod

=head1 NAME

Net::DRI::Protocol::EPP::Extensions::CentralNic::AuxContact - EPP Auxiliary Contacts Extension mapping (draft-brown-epp-auxcontact-00) for Net::DRI

=head1 DESCRIPTION

Please see the README file for details.

=head1 SUPPORT

For now, support questions should be sent to:

E<lt>netdri@dotandco.comE<gt>

Please also see the SUPPORT file in the distribution.

=head1 SEE ALSO

E<lt>http://www.dotandco.com/services/software/Net-DRI/E<gt>

=head1 AUTHOR

Patrick Mevzek, E<lt>netdri@dotandco.comE<gt>

=head1 COPYRIGHT

Copyright (c) 2016,2018 Patrick Mevzek <netdri@dotandco.com>.
All rights reserved.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

See the LICENSE file that comes with this distribution for more details.

=cut
