/*
 * Decompiled with CFR 0.152.
 */
package org.apache.distributedlog.lock;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.base.Stopwatch;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import java.util.function.Function;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.NullStatsLogger;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.util.SafeRunnable;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.distributedlog.ZooKeeperClient;
import org.apache.distributedlog.common.concurrent.FutureEventListener;
import org.apache.distributedlog.common.concurrent.FutureUtils;
import org.apache.distributedlog.common.stats.OpStatsListener;
import org.apache.distributedlog.exceptions.DLInterruptedException;
import org.apache.distributedlog.exceptions.LockingException;
import org.apache.distributedlog.exceptions.OwnershipAcquireFailedException;
import org.apache.distributedlog.exceptions.UnexpectedException;
import org.apache.distributedlog.exceptions.ZKException;
import org.apache.distributedlog.lock.DistributedLockContext;
import org.apache.distributedlog.lock.EpochChangedException;
import org.apache.distributedlog.lock.LockAction;
import org.apache.distributedlog.lock.LockClosedException;
import org.apache.distributedlog.lock.LockListener;
import org.apache.distributedlog.lock.LockSessionExpiredException;
import org.apache.distributedlog.lock.LockStateChangedException;
import org.apache.distributedlog.lock.LockWaiter;
import org.apache.distributedlog.lock.SessionLock;
import org.apache.distributedlog.util.FailpointUtils;
import org.apache.distributedlog.util.OrderedScheduler;
import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ZKSessionLock
implements SessionLock {
    static final Logger LOG = LoggerFactory.getLogger(ZKSessionLock.class);
    private static final String LOCK_PATH_PREFIX = "/member_";
    private static final String LOCK_PART_SEP = "_";
    static final Comparator<String> MEMBER_COMPARATOR = new Comparator<String>(){

        @Override
        public int compare(String o1, String o2) {
            int l1 = ZKSessionLock.parseMemberID(o1);
            int l2 = ZKSessionLock.parseMemberID(o2);
            return l1 - l2;
        }
    };
    private final ZooKeeperClient zkClient;
    private final ZooKeeper zk;
    private final String lockPath;
    private final Pair<String, Long> lockId;
    private StateManagement lockState;
    private final DistributedLockContext lockContext;
    private final CompletableFuture<Boolean> acquireFuture;
    private String currentId;
    private String currentNode;
    private String watchedNode;
    private LockWatcher watcher;
    private final AtomicInteger epoch = new AtomicInteger(0);
    private final OrderedScheduler lockStateExecutor;
    private LockListener lockListener = null;
    private final long lockOpTimeout;
    private final OpStatsLogger tryStats;
    private final Counter tryTimeouts;
    private final OpStatsLogger unlockStats;

    public static String getLockPathPrefixV1(String lockPath) {
        return lockPath + LOCK_PATH_PREFIX;
    }

    public static String getLockPathPrefixV2(String lockPath, String clientId) throws UnsupportedEncodingException {
        return lockPath + LOCK_PATH_PREFIX + URLEncoder.encode(clientId, Charsets.UTF_8.name()) + LOCK_PART_SEP;
    }

    public static String getLockPathPrefixV3(String lockPath, String clientId, long sessionOwner) throws UnsupportedEncodingException {
        StringBuilder sb = new StringBuilder();
        sb.append(lockPath).append(LOCK_PATH_PREFIX).append(URLEncoder.encode(clientId, Charsets.UTF_8.name())).append(LOCK_PART_SEP).append("s").append(String.format("%10d", sessionOwner)).append(LOCK_PART_SEP);
        return sb.toString();
    }

    public static byte[] serializeClientId(String clientId) {
        return clientId.getBytes(Charsets.UTF_8);
    }

    public static String deserializeClientId(byte[] data) {
        return new String(data, Charsets.UTF_8);
    }

    public static String getLockIdFromPath(String path) {
        if (path.contains("/")) {
            return path.substring(path.lastIndexOf("/") + 1);
        }
        return path;
    }

    ZKSessionLock(ZooKeeperClient zkClient, String lockPath, String clientId, OrderedScheduler lockStateExecutor) throws IOException {
        this(zkClient, lockPath, clientId, lockStateExecutor, 120000L, (StatsLogger)NullStatsLogger.INSTANCE, new DistributedLockContext());
    }

    public ZKSessionLock(ZooKeeperClient zkClient, String lockPath, String clientId, OrderedScheduler lockStateExecutor, long lockOpTimeout, StatsLogger statsLogger, DistributedLockContext lockContext) throws IOException {
        this.zkClient = zkClient;
        try {
            this.zk = zkClient.get();
        }
        catch (ZooKeeperClient.ZooKeeperConnectionException zce) {
            throw new ZKException("Failed to get zookeeper client for lock " + lockPath, KeeperException.Code.CONNECTIONLOSS);
        }
        catch (InterruptedException e) {
            throw new DLInterruptedException("Interrupted on getting zookeeper client for lock " + lockPath, (Throwable)e);
        }
        this.lockPath = lockPath;
        this.lockId = Pair.of((Object)clientId, (Object)this.zk.getSessionId());
        this.lockContext = lockContext;
        this.lockStateExecutor = lockStateExecutor;
        this.lockState = new StateManagement();
        this.lockOpTimeout = lockOpTimeout;
        this.tryStats = statsLogger.getOpStatsLogger("tryAcquire");
        this.tryTimeouts = statsLogger.getCounter("tryTimeouts");
        this.unlockStats = statsLogger.getOpStatsLogger("unlock");
        this.acquireFuture = FutureUtils.createFuture();
        this.acquireFuture.whenComplete((value, cause) -> {
            if (null != cause) {
                this.asyncUnlock((Throwable)cause);
            }
        });
    }

    @Override
    public ZKSessionLock setLockListener(LockListener lockListener) {
        this.lockListener = lockListener;
        return this;
    }

    String getLockPath() {
        return this.lockPath;
    }

    @VisibleForTesting
    AtomicInteger getEpoch() {
        return this.epoch;
    }

    @VisibleForTesting
    State getLockState() {
        return this.lockState.getState();
    }

    @VisibleForTesting
    Pair<String, Long> getLockId() {
        return this.lockId;
    }

    @Override
    public boolean isLockExpired() {
        return this.lockState.isExpiredOrClosing();
    }

    @Override
    public boolean isLockHeld() {
        return this.lockState.inState(State.CLAIMED);
    }

    protected void executeLockAction(final int lockEpoch, final LockAction func) {
        this.lockStateExecutor.submit((Object)this.lockPath, (Runnable)new SafeRunnable(){

            public void safeRun() {
                if (ZKSessionLock.this.epoch.get() == lockEpoch) {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("{} executing lock action '{}' under epoch {} for lock {}", new Object[]{ZKSessionLock.this.lockId, func.getActionName(), lockEpoch, ZKSessionLock.this.lockPath});
                    }
                    func.execute();
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("{} executed lock action '{}' under epoch {} for lock {}", new Object[]{ZKSessionLock.this.lockId, func.getActionName(), lockEpoch, ZKSessionLock.this.lockPath});
                    }
                } else if (LOG.isTraceEnabled()) {
                    LOG.trace("{} skipped executing lock action '{}' for lock {}, since epoch is changed from {} to {}.", new Object[]{ZKSessionLock.this.lockId, func.getActionName(), ZKSessionLock.this.lockPath, lockEpoch, ZKSessionLock.this.epoch.get()});
                }
            }
        });
    }

    protected <T> void executeLockAction(final int lockEpoch, final LockAction func, final CompletableFuture<T> promise) {
        this.lockStateExecutor.submit((Object)this.lockPath, (Runnable)new SafeRunnable(){

            public void safeRun() {
                int currentEpoch = ZKSessionLock.this.epoch.get();
                if (currentEpoch == lockEpoch) {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("{} executed lock action '{}' under epoch {} for lock {}", new Object[]{ZKSessionLock.this.lockId, func.getActionName(), lockEpoch, ZKSessionLock.this.lockPath});
                    }
                    func.execute();
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("{} executed lock action '{}' under epoch {} for lock {}", new Object[]{ZKSessionLock.this.lockId, func.getActionName(), lockEpoch, ZKSessionLock.this.lockPath});
                    }
                } else {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("{} skipped executing lock action '{}' for lock {}, since epoch is changed from {} to {}.", new Object[]{ZKSessionLock.this.lockId, func.getActionName(), ZKSessionLock.this.lockPath, lockEpoch, currentEpoch});
                    }
                    promise.completeExceptionally((Throwable)((Object)new EpochChangedException(ZKSessionLock.this.lockPath, lockEpoch, currentEpoch)));
                }
            }
        });
    }

    static int parseMemberID(String nodeName) {
        int id = -1;
        String[] parts = nodeName.split(LOCK_PART_SEP);
        if (parts.length > 0) {
            try {
                id = Integer.parseInt(parts[parts.length - 1]);
            }
            catch (NumberFormatException nfe) {
                id = Integer.MAX_VALUE;
            }
        }
        return id;
    }

    static boolean areLockWaitersInSameSession(String node1, String node2) {
        long sessionOwner2;
        String[] parts1 = node1.split(LOCK_PART_SEP);
        String[] parts2 = node2.split(LOCK_PART_SEP);
        if (parts1.length != 4 || parts2.length != 4) {
            return node1.equals(node2);
        }
        if (!parts1[2].startsWith("s") || !parts2[2].startsWith("s")) {
            return node1.equals(node2);
        }
        long sessionOwner1 = Long.parseLong(parts1[2].substring(1));
        if (sessionOwner1 != (sessionOwner2 = Long.parseLong(parts2[2].substring(1)))) {
            return false;
        }
        try {
            String clientId1 = URLDecoder.decode(parts1[1], Charsets.UTF_8.name());
            String clientId2 = URLDecoder.decode(parts2[1], Charsets.UTF_8.name());
            return clientId1.equals(clientId2);
        }
        catch (UnsupportedEncodingException e) {
            return node1.equals(node2);
        }
    }

    static CompletableFuture<Pair<String, Long>> asyncParseClientID(ZooKeeper zkClient, String lockPath, String nodeName) {
        String[] parts = nodeName.split(LOCK_PART_SEP);
        if (4 == parts.length && parts[2].startsWith("s")) {
            long sessionOwner = Long.parseLong(parts[2].substring(1));
            try {
                String clientId = URLDecoder.decode(parts[1], Charsets.UTF_8.name());
                return FutureUtils.value((Object)Pair.of((Object)clientId, (Object)sessionOwner));
            }
            catch (UnsupportedEncodingException unsupportedEncodingException) {
                // empty catch block
            }
        }
        final CompletableFuture<Pair<String, Long>> promise = new CompletableFuture<Pair<String, Long>>();
        zkClient.getData(lockPath + "/" + nodeName, false, new AsyncCallback.DataCallback(){

            public void processResult(int rc, String path, Object ctx, byte[] data, Stat stat) {
                if (KeeperException.Code.OK.intValue() != rc) {
                    promise.completeExceptionally(KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc)));
                } else {
                    promise.complete(Pair.of((Object)ZKSessionLock.deserializeClientId(data), (Object)stat.getEphemeralOwner()));
                }
            }
        }, null);
        return promise;
    }

    @Override
    public CompletableFuture<LockWaiter> asyncTryLock(final long timeout, final TimeUnit unit) {
        boolean wait;
        final CompletableFuture<String> result = new CompletableFuture<String>();
        boolean bl = wait = 0L != timeout;
        if (wait) {
            this.asyncTryLock(wait, result);
        } else {
            this.zk.getChildren(this.lockPath, null, new AsyncCallback.Children2Callback(){

                public void processResult(final int rc, String path, Object ctx, final List<String> children, Stat stat) {
                    ZKSessionLock.this.lockStateExecutor.submit((Object)ZKSessionLock.this.lockPath, (Runnable)new SafeRunnable(){

                        public void safeRun() {
                            if (!ZKSessionLock.this.lockState.inState(State.INIT)) {
                                result.completeExceptionally((Throwable)((Object)new LockStateChangedException(ZKSessionLock.this.lockPath, (Pair<String, Long>)ZKSessionLock.this.lockId, State.INIT, ZKSessionLock.this.lockState.getState())));
                                return;
                            }
                            if (KeeperException.Code.OK.intValue() != rc) {
                                result.completeExceptionally(KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc)));
                                return;
                            }
                            FailpointUtils.checkFailPointNoThrow(FailpointUtils.FailPointName.FP_LockTryAcquire);
                            Collections.sort(children, MEMBER_COMPARATOR);
                            if (children.size() > 0) {
                                ZKSessionLock.asyncParseClientID(ZKSessionLock.this.zk, ZKSessionLock.this.lockPath, (String)children.get(0)).whenCompleteAsync((BiConsumer)new FutureEventListener<Pair<String, Long>>(){

                                    public void onSuccess(Pair<String, Long> owner) {
                                        if (!ZKSessionLock.this.checkOrClaimLockOwner((Pair<String, Long>)owner, result)) {
                                            ZKSessionLock.this.acquireFuture.complete(false);
                                        }
                                    }

                                    public void onFailure(Throwable cause) {
                                        result.completeExceptionally(cause);
                                    }
                                }, (Executor)ZKSessionLock.this.lockStateExecutor.chooseExecutor((Object)ZKSessionLock.this.lockPath));
                            } else {
                                ZKSessionLock.this.asyncTryLock(wait, result);
                            }
                        }
                    });
                }
            }, null);
        }
        final CompletableFuture waiterAcquireFuture = FutureUtils.createFuture();
        waiterAcquireFuture.whenComplete((value, cause) -> this.acquireFuture.completeExceptionally((Throwable)cause));
        return result.thenApply(new Function<String, LockWaiter>(){

            @Override
            public LockWaiter apply(String currentOwner) {
                OwnershipAcquireFailedException acquireException = new OwnershipAcquireFailedException(ZKSessionLock.this.lockPath, currentOwner);
                FutureUtils.within((CompletableFuture)ZKSessionLock.this.acquireFuture, (long)timeout, (TimeUnit)unit, (Throwable)acquireException, (OrderedScheduler)ZKSessionLock.this.lockStateExecutor, (Object)ZKSessionLock.this.lockPath).whenComplete((BiConsumer)new FutureEventListener<Boolean>((Exception)acquireException){
                    final /* synthetic */ Exception val$acquireException;
                    {
                        this.val$acquireException = exception;
                    }

                    public void onSuccess(Boolean acquired) {
                        this.completeOrFail(this.val$acquireException);
                    }

                    public void onFailure(Throwable acquireCause) {
                        this.completeOrFail(this.val$acquireException);
                    }

                    private void completeOrFail(final Throwable acquireCause) {
                        if (ZKSessionLock.this.isLockHeld()) {
                            waiterAcquireFuture.complete(true);
                        } else {
                            ZKSessionLock.this.asyncUnlock().whenComplete((BiConsumer)new FutureEventListener<Void>(){

                                public void onSuccess(Void value) {
                                    waiterAcquireFuture.completeExceptionally(acquireCause);
                                }

                                public void onFailure(Throwable cause) {
                                    waiterAcquireFuture.completeExceptionally(acquireCause);
                                }
                            });
                        }
                    }
                });
                return new LockWaiter((String)ZKSessionLock.this.lockId.getLeft(), currentOwner, waiterAcquireFuture);
            }
        });
    }

    private boolean checkOrClaimLockOwner(final Pair<String, Long> currentOwner, final CompletableFuture<String> result) {
        if (this.lockId.compareTo(currentOwner) != 0 && !this.lockContext.hasLockId(currentOwner)) {
            this.lockStateExecutor.submit((Object)this.lockPath, (Runnable)new SafeRunnable(){

                public void safeRun() {
                    result.complete(currentOwner.getLeft());
                }
            });
            return false;
        }
        int curEpoch = this.epoch.incrementAndGet();
        this.executeLockAction(curEpoch, new LockAction(){

            @Override
            public void execute() {
                if (!ZKSessionLock.this.lockState.inState(State.INIT)) {
                    result.completeExceptionally((Throwable)((Object)new LockStateChangedException(ZKSessionLock.this.lockPath, (Pair<String, Long>)ZKSessionLock.this.lockId, State.INIT, ZKSessionLock.this.lockState.getState())));
                    return;
                }
                ZKSessionLock.this.asyncTryLock(false, result);
            }

            @Override
            public String getActionName() {
                return "claimOwnership(owner=" + currentOwner + ")";
            }
        }, result);
        return true;
    }

    private void asyncTryLock(boolean wait, final CompletableFuture<String> result) {
        CompletableFuture<String> lockResult = new CompletableFuture<String>();
        lockResult.whenComplete((BiConsumer)new FutureEventListener<String>(){

            public void onSuccess(String currentOwner) {
                result.complete(currentOwner);
            }

            public void onFailure(final Throwable lockCause) {
                if (lockCause instanceof LockStateChangedException) {
                    LOG.info("skipping cleanup for {} at {} after encountering lock state change exception : ", new Object[]{ZKSessionLock.this.lockId, ZKSessionLock.this.lockPath, lockCause});
                    result.completeExceptionally(lockCause);
                    return;
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{} is cleaning up its lock state for {} due to : ", new Object[]{ZKSessionLock.this.lockId, ZKSessionLock.this.lockPath, lockCause});
                }
                CompletableFuture<Void> unlockResult = ZKSessionLock.this.asyncUnlock();
                unlockResult.whenComplete((BiConsumer)new FutureEventListener<Void>(){

                    public void onSuccess(Void value) {
                        result.completeExceptionally(lockCause);
                    }

                    public void onFailure(Throwable cause) {
                        result.completeExceptionally(lockCause);
                    }
                });
            }
        });
        this.asyncTryLockWithoutCleanup(wait, lockResult);
    }

    private void asyncTryLockWithoutCleanup(final boolean wait, final CompletableFuture<String> promise) {
        this.executeLockAction(this.epoch.get(), new LockAction(){

            @Override
            public void execute() {
                String myPath;
                if (!ZKSessionLock.this.lockState.inState(State.INIT)) {
                    promise.completeExceptionally((Throwable)((Object)new LockStateChangedException(ZKSessionLock.this.lockPath, (Pair<String, Long>)ZKSessionLock.this.lockId, State.INIT, ZKSessionLock.this.lockState.getState())));
                    return;
                }
                ZKSessionLock.this.lockState.transition(State.PREPARING);
                final int curEpoch = ZKSessionLock.this.epoch.incrementAndGet();
                ZKSessionLock.this.watcher = new LockWatcher(curEpoch);
                ZKSessionLock.this.zkClient.register(ZKSessionLock.this.watcher);
                try {
                    myPath = ZKSessionLock.getLockPathPrefixV3(ZKSessionLock.this.lockPath, (String)ZKSessionLock.this.lockId.getLeft(), (Long)ZKSessionLock.this.lockId.getRight());
                }
                catch (UnsupportedEncodingException uee) {
                    myPath = ZKSessionLock.getLockPathPrefixV1(ZKSessionLock.this.lockPath);
                }
                ZKSessionLock.this.zk.create(myPath, ZKSessionLock.serializeClientId((String)ZKSessionLock.this.lockId.getLeft()), ZKSessionLock.this.zkClient.getDefaultACL(), CreateMode.EPHEMERAL_SEQUENTIAL, new AsyncCallback.StringCallback(){

                    public void processResult(final int rc, String path, Object ctx, final String name) {
                        ZKSessionLock.this.executeLockAction(curEpoch, new LockAction(){

                            @Override
                            public void execute() {
                                if (KeeperException.Code.OK.intValue() != rc) {
                                    KeeperException ke = KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc));
                                    promise.completeExceptionally(ke);
                                    return;
                                }
                                if (FailpointUtils.checkFailPointNoThrow(FailpointUtils.FailPointName.FP_LockTryCloseRaceCondition)) {
                                    ZKSessionLock.this.lockState.transition(State.CLOSING);
                                    ZKSessionLock.this.lockState.transition(State.CLOSED);
                                }
                                if (null != ZKSessionLock.this.currentNode) {
                                    LOG.error("Current node for {} overwritten current = {} new = {}", new Object[]{ZKSessionLock.this.lockPath, ZKSessionLock.this.lockId, ZKSessionLock.getLockIdFromPath(ZKSessionLock.this.currentNode)});
                                }
                                ZKSessionLock.this.currentNode = name;
                                ZKSessionLock.this.currentId = ZKSessionLock.getLockIdFromPath(ZKSessionLock.this.currentNode);
                                LOG.trace("{} received member id for lock {}", (Object)ZKSessionLock.this.lockId, (Object)ZKSessionLock.this.currentId);
                                if (ZKSessionLock.this.lockState.isExpiredOrClosing()) {
                                    CompletableFuture deletePromise = new CompletableFuture();
                                    ZKSessionLock.this.deleteLockNode(deletePromise);
                                    FutureUtils.ensure(deletePromise, () -> promise.completeExceptionally((Throwable)((Object)new LockClosedException(ZKSessionLock.this.lockPath, (Pair<String, Long>)ZKSessionLock.this.lockId, ZKSessionLock.this.lockState.getState()))));
                                    return;
                                }
                                ZKSessionLock.this.lockState.transition(State.PREPARED);
                                ZKSessionLock.this.checkLockOwnerAndWaitIfPossible(ZKSessionLock.this.watcher, wait, promise);
                            }

                            @Override
                            public String getActionName() {
                                return "postPrepare(wait=" + wait + ")";
                            }
                        });
                    }
                }, null);
            }

            @Override
            public String getActionName() {
                return "prepare(wait=" + wait + ")";
            }
        }, promise);
    }

    @Override
    public void tryLock(long timeout, TimeUnit unit) throws LockingException {
        CompletableFuture<LockWaiter> tryFuture;
        Stopwatch stopwatch = Stopwatch.createStarted();
        LockWaiter waiter = this.waitForTry(stopwatch, tryFuture = this.asyncTryLock(timeout, unit));
        boolean acquired = waiter.waitForAcquireQuietly();
        if (!acquired) {
            throw new OwnershipAcquireFailedException(this.lockPath, waiter.getCurrentOwner());
        }
    }

    synchronized LockWaiter waitForTry(Stopwatch stopwatch, CompletableFuture<LockWaiter> tryFuture) throws LockingException {
        LockWaiter waiter;
        boolean success = false;
        boolean stateChanged = false;
        try {
            waiter = (LockWaiter)FutureUtils.result(tryFuture, (long)this.lockOpTimeout, (TimeUnit)TimeUnit.MILLISECONDS);
            success = true;
        }
        catch (LockStateChangedException ex) {
            stateChanged = true;
            throw ex;
        }
        catch (LockingException ex) {
            throw ex;
        }
        catch (TimeoutException toe) {
            this.tryTimeouts.inc();
            throw new LockingException(this.lockPath, "Timeout during try phase of lock acquire", (Throwable)toe);
        }
        catch (Exception ex) {
            String message = this.getLockId() + " failed to lock " + this.lockPath;
            throw new LockingException(this.lockPath, message, (Throwable)ex);
        }
        finally {
            if (success) {
                this.tryStats.registerSuccessfulEvent(stopwatch.elapsed(TimeUnit.MICROSECONDS), TimeUnit.MICROSECONDS);
            } else {
                this.tryStats.registerFailedEvent(stopwatch.elapsed(TimeUnit.MICROSECONDS), TimeUnit.MICROSECONDS);
            }
            if (!success && !stateChanged) {
                this.unlock();
            }
        }
        return waiter;
    }

    @Override
    public CompletableFuture<Void> asyncUnlock() {
        return this.asyncUnlock((Throwable)((Object)new LockClosedException(this.lockPath, this.lockId, this.lockState.getState())));
    }

    CompletableFuture<Void> asyncUnlock(final Throwable cause) {
        final CompletableFuture<Void> promise = new CompletableFuture<Void>();
        this.lockStateExecutor.submit((Object)this.lockPath, (Runnable)new SafeRunnable(){

            public void safeRun() {
                ZKSessionLock.this.acquireFuture.completeExceptionally(cause);
                ZKSessionLock.this.unlockInternal(promise);
                promise.whenComplete((BiConsumer)new OpStatsListener(ZKSessionLock.this.unlockStats));
            }
        });
        return promise;
    }

    @Override
    public void unlock() {
        CompletableFuture<Void> unlockResult = this.asyncUnlock();
        try {
            FutureUtils.result(unlockResult, (long)this.lockOpTimeout, (TimeUnit)TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException toe) {
            LOG.error("Timeout unlocking {} owned by {} : ", new Object[]{this.lockPath, this.lockId, toe});
        }
        catch (Exception e) {
            LOG.warn("{} failed to unlock {} : ", new Object[]{this.lockId, this.lockPath, e});
        }
    }

    private void claimOwnership(int lockEpoch) {
        this.lockState.transition(State.CLAIMED);
        this.lockContext.clearLockIds();
        this.lockContext.addLockId(this.lockId);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Notify lock waiters on {} at {} : watcher epoch {}, lock epoch {}", new Object[]{this.lockPath, System.currentTimeMillis(), lockEpoch, this.epoch.get()});
        }
        this.acquireFuture.complete(true);
    }

    private void unlockInternal(final CompletableFuture<Void> promise) {
        this.epoch.incrementAndGet();
        if (null != this.watcher) {
            this.zkClient.unregister(this.watcher);
        }
        if (this.lockState.inState(State.CLOSED)) {
            promise.complete(null);
            return;
        }
        LOG.info("Lock {} for {} is closed from state {}.", new Object[]{this.lockId, this.lockPath, this.lockState.getState()});
        boolean skipCleanup = this.lockState.inState(State.INIT) || this.lockState.inState(State.EXPIRED);
        this.lockState.transition(State.CLOSING);
        if (skipCleanup) {
            this.lockState.transition(State.CLOSED);
            promise.complete(null);
            return;
        }
        CompletableFuture<Void> deletePromise = new CompletableFuture<Void>();
        this.deleteLockNode(deletePromise);
        deletePromise.whenCompleteAsync((BiConsumer)new FutureEventListener<Void>(){

            public void onSuccess(Void complete) {
                ZKSessionLock.this.lockState.transition(State.CLOSED);
                promise.complete(null);
            }

            public void onFailure(Throwable cause) {
                LOG.error("lock node delete failed {} {}", (Object)ZKSessionLock.this.lockId, (Object)ZKSessionLock.this.lockPath);
                promise.complete(null);
            }
        }, (Executor)this.lockStateExecutor.chooseExecutor((Object)this.lockPath));
    }

    private void deleteLockNode(final CompletableFuture<Void> promise) {
        if (null == this.currentNode) {
            promise.complete(null);
            return;
        }
        this.zk.delete(this.currentNode, -1, new AsyncCallback.VoidCallback(){

            public void processResult(final int rc, final String path, Object ctx) {
                ZKSessionLock.this.lockStateExecutor.submit((Object)ZKSessionLock.this.lockPath, (Runnable)new SafeRunnable(){

                    public void safeRun() {
                        if (KeeperException.Code.OK.intValue() == rc) {
                            LOG.info("Deleted lock node {} for {} successfully.", (Object)path, (Object)ZKSessionLock.this.lockId);
                        } else if (KeeperException.Code.NONODE.intValue() == rc || KeeperException.Code.SESSIONEXPIRED.intValue() == rc) {
                            LOG.info("Delete node failed. Node already gone for node {} id {}, rc = {}", new Object[]{path, ZKSessionLock.this.lockId, KeeperException.Code.get((int)rc)});
                        } else {
                            LOG.error("Failed on deleting lock node {} for {} : {}", new Object[]{path, ZKSessionLock.this.lockId, KeeperException.Code.get((int)rc)});
                        }
                        FailpointUtils.checkFailPointNoThrow(FailpointUtils.FailPointName.FP_LockUnlockCleanup);
                        promise.complete(null);
                    }
                });
            }
        }, null);
    }

    private void handleSessionExpired(final int lockEpoch) {
        this.executeLockAction(lockEpoch, new LockAction(){

            @Override
            public void execute() {
                if (ZKSessionLock.this.lockState.inState(State.CLOSED) || ZKSessionLock.this.lockState.inState(State.CLOSING)) {
                    return;
                }
                boolean shouldNotifyLockListener = ZKSessionLock.this.lockState.inState(State.CLAIMED);
                ZKSessionLock.this.lockState.transition(State.EXPIRED);
                if (null != ZKSessionLock.this.watcher) {
                    ZKSessionLock.this.zkClient.unregister(ZKSessionLock.this.watcher);
                }
                ZKSessionLock.this.epoch.incrementAndGet();
                ZKSessionLock.this.acquireFuture.completeExceptionally((Throwable)((Object)new LockSessionExpiredException(ZKSessionLock.this.lockPath, (Pair<String, Long>)ZKSessionLock.this.lockId, ZKSessionLock.this.lockState.getState())));
                ZKSessionLock.this.currentNode = null;
                ZKSessionLock.this.currentId = null;
                if (shouldNotifyLockListener && null != ZKSessionLock.this.lockListener) {
                    ZKSessionLock.this.lockListener.onExpired();
                }
            }

            @Override
            public String getActionName() {
                return "handleSessionExpired(epoch=" + lockEpoch + ")";
            }
        });
    }

    private void handleNodeDelete(int lockEpoch, final WatchedEvent event) {
        this.executeLockAction(lockEpoch, new LockAction(){

            @Override
            public void execute() {
                if (!ZKSessionLock.this.lockState.inState(State.WAITING)) {
                    LOG.info("{} ignore watched node {} deleted event, since lock state has moved to {}.", new Object[]{ZKSessionLock.this.lockId, event.getPath(), ZKSessionLock.this.lockState.getState()});
                    return;
                }
                ZKSessionLock.this.lockState.transition(State.PREPARED);
                ZKSessionLock.this.checkLockOwnerAndWaitIfPossible(ZKSessionLock.this.watcher, true);
            }

            @Override
            public String getActionName() {
                return "handleNodeDelete(path=" + event.getPath() + ")";
            }
        });
    }

    private CompletableFuture<String> checkLockOwnerAndWaitIfPossible(LockWatcher lockWatcher, boolean wait) {
        CompletableFuture<String> promise = new CompletableFuture<String>();
        this.checkLockOwnerAndWaitIfPossible(lockWatcher, wait, promise);
        return promise;
    }

    private void checkLockOwnerAndWaitIfPossible(final LockWatcher lockWatcher, final boolean wait, final CompletableFuture<String> promise) {
        this.zk.getChildren(this.lockPath, false, new AsyncCallback.Children2Callback(){

            public void processResult(int rc, String path, Object ctx, List<String> children, Stat stat) {
                ZKSessionLock.this.processLockWaiters(lockWatcher, wait, rc, children, promise);
            }
        }, null);
    }

    private void processLockWaiters(final LockWatcher lockWatcher, final boolean wait, final int getChildrenRc, final List<String> children, final CompletableFuture<String> promise) {
        this.executeLockAction(lockWatcher.epoch, new LockAction(){

            @Override
            public void execute() {
                if (!ZKSessionLock.this.lockState.inState(State.PREPARED)) {
                    promise.completeExceptionally((Throwable)((Object)new LockStateChangedException(ZKSessionLock.this.lockPath, (Pair<String, Long>)ZKSessionLock.this.lockId, State.PREPARED, ZKSessionLock.this.lockState.getState())));
                    return;
                }
                if (KeeperException.Code.OK.intValue() != getChildrenRc) {
                    promise.completeExceptionally(KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)getChildrenRc)));
                    return;
                }
                if (children.isEmpty()) {
                    LOG.error("Error, member list is empty for lock {}.", (Object)ZKSessionLock.this.lockPath);
                    promise.completeExceptionally(new UnexpectedException("Empty member list for lock " + ZKSessionLock.this.lockPath));
                    return;
                }
                Collections.sort(children, MEMBER_COMPARATOR);
                final String cid = ZKSessionLock.this.currentId;
                final int memberIndex = children.indexOf(cid);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{} is the number {} member in the list.", (Object)cid, (Object)memberIndex);
                }
                if (memberIndex == 0) {
                    LOG.info("{} acquired the lock {}.", (Object)cid, (Object)ZKSessionLock.this.lockPath);
                    ZKSessionLock.this.claimOwnership(lockWatcher.epoch);
                    promise.complete(cid);
                } else if (memberIndex > 0) {
                    ZKSessionLock.asyncParseClientID(ZKSessionLock.this.zk, ZKSessionLock.this.lockPath, (String)children.get(0)).whenComplete((BiConsumer)new FutureEventListener<Pair<String, Long>>(){

                        public void onSuccess(Pair<String, Long> currentOwner) {
                            ZKSessionLock.this.watchLockOwner(lockWatcher, wait, cid, (String)children.get(memberIndex - 1), (String)children.get(0), (Pair<String, Long>)currentOwner, promise);
                        }

                        public void onFailure(final Throwable cause) {
                            ZKSessionLock.this.executeLockAction(lockWatcher.epoch, new LockAction(){

                                @Override
                                public void execute() {
                                    promise.completeExceptionally(cause);
                                }

                                @Override
                                public String getActionName() {
                                    return "handleFailureOnParseClientID(lockPath=" + ZKSessionLock.this.lockPath + ")";
                                }
                            }, promise);
                        }
                    });
                } else {
                    LOG.error("Member {} doesn't exist in the members list {} for lock {}.", new Object[]{cid, children, ZKSessionLock.this.lockPath});
                    promise.completeExceptionally(new UnexpectedException("Member " + cid + " doesn't exist in member list " + children + " for lock " + ZKSessionLock.this.lockPath));
                }
            }

            @Override
            public String getActionName() {
                return "processLockWaiters(rc=" + getChildrenRc + ", waiters=" + children + ")";
            }
        }, promise);
    }

    private void watchLockOwner(final LockWatcher lockWatcher, final boolean wait, final String myNode, final String siblingNode, final String ownerNode, final Pair<String, Long> currentOwner, final CompletableFuture<String> promise) {
        this.executeLockAction(lockWatcher.epoch, new LockAction(){

            @Override
            public void execute() {
                boolean shouldClaimOwnership;
                boolean shouldWatch;
                if (ZKSessionLock.this.lockContext.hasLockId((Pair<String, Long>)currentOwner) && siblingNode.equals(ownerNode)) {
                    shouldWatch = true;
                    shouldClaimOwnership = true;
                    LOG.info("LockWatcher {} for {} found its previous session {} held lock, watch it to claim ownership.", new Object[]{myNode, ZKSessionLock.this.lockPath, currentOwner});
                } else if (ZKSessionLock.this.lockId.compareTo(currentOwner) == 0 && ZKSessionLock.areLockWaitersInSameSession(siblingNode, ownerNode)) {
                    shouldWatch = true;
                    shouldClaimOwnership = true;
                    LOG.info("LockWatcher {} for {} found itself {} already held lock at sibling node {}, watch it to claim ownership.", new Object[]{myNode, ZKSessionLock.this.lockPath, ZKSessionLock.this.lockId, siblingNode});
                } else {
                    shouldWatch = wait;
                    if (wait && LOG.isDebugEnabled()) {
                        LOG.debug("Current LockWatcher for {} with ephemeral node {}, is waiting for {} to release lock at {}.", new Object[]{ZKSessionLock.this.lockPath, myNode, siblingNode, System.currentTimeMillis()});
                    }
                    shouldClaimOwnership = false;
                }
                if (shouldWatch) {
                    ZKSessionLock.this.watchedNode = String.format("%s/%s", ZKSessionLock.this.lockPath, siblingNode);
                    ZKSessionLock.this.zk.exists(ZKSessionLock.this.watchedNode, (Watcher)lockWatcher, new AsyncCallback.StatCallback(){

                        public void processResult(final int rc, String path, Object ctx, Stat stat) {
                            ZKSessionLock.this.executeLockAction(lockWatcher.epoch, new LockAction(){

                                @Override
                                public void execute() {
                                    if (!ZKSessionLock.this.lockState.inState(State.PREPARED)) {
                                        promise.completeExceptionally((Throwable)((Object)new LockStateChangedException(ZKSessionLock.this.lockPath, (Pair<String, Long>)ZKSessionLock.this.lockId, State.PREPARED, ZKSessionLock.this.lockState.getState())));
                                        return;
                                    }
                                    if (KeeperException.Code.OK.intValue() == rc) {
                                        if (shouldClaimOwnership) {
                                            LOG.info("LockWatcher {} claimed ownership for {} after set watcher on {}.", new Object[]{myNode, ZKSessionLock.this.lockPath, ownerNode});
                                            ZKSessionLock.this.claimOwnership(lockWatcher.epoch);
                                            promise.complete(currentOwner.getLeft());
                                        } else {
                                            ZKSessionLock.this.lockState.transition(State.WAITING);
                                            promise.complete(currentOwner.getLeft());
                                        }
                                    } else if (KeeperException.Code.NONODE.intValue() == rc) {
                                        ZKSessionLock.this.checkLockOwnerAndWaitIfPossible(lockWatcher, wait, promise);
                                    } else {
                                        promise.completeExceptionally(KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc)));
                                    }
                                }

                                @Override
                                public String getActionName() {
                                    StringBuilder sb = new StringBuilder();
                                    sb.append("postWatchLockOwner(myNode=").append(myNode).append(", siblingNode=").append(siblingNode).append(", ownerNode=").append(ownerNode).append(")");
                                    return sb.toString();
                                }
                            }, promise);
                        }
                    }, null);
                } else {
                    promise.complete(currentOwner.getLeft());
                }
            }

            @Override
            public String getActionName() {
                StringBuilder sb = new StringBuilder();
                sb.append("watchLockOwner(myNode=").append(myNode).append(", siblingNode=").append(siblingNode).append(", ownerNode=").append(ownerNode).append(")");
                return sb.toString();
            }
        }, promise);
    }

    class LockWatcher
    implements Watcher {
        final int epoch;

        LockWatcher(int epoch) {
            this.epoch = epoch;
        }

        public void process(WatchedEvent event) {
            LOG.debug("Received event {} from lock {} at {} : watcher epoch {}, lock epoch {}.", new Object[]{event, ZKSessionLock.this.lockPath, System.currentTimeMillis(), this.epoch, ZKSessionLock.this.epoch.get()});
            if (event.getType() == Watcher.Event.EventType.None) {
                switch (event.getState()) {
                    case SyncConnected: {
                        break;
                    }
                    case Expired: {
                        LOG.info("Session {} is expired for lock {} at {} : watcher epoch {}, lock epoch {}.", new Object[]{ZKSessionLock.this.lockId.getRight(), ZKSessionLock.this.lockPath, System.currentTimeMillis(), this.epoch, ZKSessionLock.this.epoch.get()});
                        ZKSessionLock.this.handleSessionExpired(this.epoch);
                        break;
                    }
                }
            } else if (event.getType() == Watcher.Event.EventType.NodeDeleted) {
                if (!event.getPath().equals(ZKSessionLock.this.watchedNode)) {
                    LOG.warn("{} (watching {}) ignored watched event from {} ", new Object[]{ZKSessionLock.this.lockId, ZKSessionLock.this.watchedNode, event.getPath()});
                    return;
                }
                ZKSessionLock.this.handleNodeDelete(this.epoch, event);
            } else {
                LOG.warn("Unexpected ZK event: {}", (Object)event.getType().name());
            }
        }
    }

    static class StateManagement {
        static final Logger LOG = LoggerFactory.getLogger(StateManagement.class);
        private volatile State state = State.INIT;

        StateManagement() {
        }

        public void transition(State toState) {
            if (!this.validTransition(toState)) {
                LOG.error("Invalid state transition from {} to {} ", new Object[]{this.state, toState, this.getStack()});
            }
            this.state = toState;
        }

        private boolean validTransition(State toState) {
            switch (toState) {
                case INIT: {
                    return false;
                }
                case PREPARING: {
                    return this.inState(State.INIT);
                }
                case PREPARED: {
                    return this.inState(State.PREPARING) || this.inState(State.WAITING);
                }
                case CLAIMED: {
                    return this.inState(State.PREPARED);
                }
                case WAITING: {
                    return this.inState(State.PREPARED);
                }
                case EXPIRED: {
                    return this.isTryingOrClaimed();
                }
                case CLOSING: {
                    return !this.inState(State.CLOSED);
                }
                case CLOSED: {
                    return this.inState(State.CLOSING) || this.inState(State.CLOSED);
                }
            }
            return false;
        }

        private State getState() {
            return this.state;
        }

        private boolean isTryingOrClaimed() {
            return this.inState(State.PREPARING) || this.inState(State.PREPARED) || this.inState(State.WAITING) || this.inState(State.CLAIMED);
        }

        public boolean isExpiredOrClosing() {
            return this.inState(State.CLOSED) || this.inState(State.EXPIRED) || this.inState(State.CLOSING);
        }

        public boolean isExpiredOrClosed() {
            return this.inState(State.CLOSED) || this.inState(State.EXPIRED);
        }

        public boolean isClosed() {
            return this.inState(State.CLOSED);
        }

        private boolean inState(State state) {
            return state == this.state;
        }

        private Exception getStack() {
            return new Exception();
        }
    }

    static enum State {
        INIT,
        PREPARING,
        PREPARED,
        CLAIMED,
        WAITING,
        EXPIRED,
        CLOSING,
        CLOSED;

    }
}

