/*
 * Decompiled with CFR 0.152.
 */
package org.apache.drill.yarn.appMaster;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.drill.yarn.appMaster.EventContext;
import org.apache.drill.yarn.appMaster.Task;
import org.apache.drill.yarn.appMaster.TaskLifecycleListener;
import org.apache.drill.yarn.appMaster.YarnFacadeException;
import org.apache.drill.yarn.core.DoYUtil;
import org.apache.hadoop.yarn.api.records.Container;
import org.apache.hadoop.yarn.api.records.ContainerStatus;

public abstract class TaskState {
    private static final Log LOG = LogFactory.getLog(TaskState.class);
    public static final TaskState START = new StartState();
    public static final TaskState REQUESTING = new RequestingState();
    public static final TaskState LAUNCHING = new LaunchingState();
    public static final TaskState WAIT_START_ACK = new WaitStartAckState();
    public static final TaskState RUNNING = new RunningState();
    public static final TaskState ENDING = new EndingState();
    public static final TaskState KILLING = new KillingState();
    public static final TaskState WAIT_END_ACK = new WaitEndAckState();
    public static final TaskState END = new EndState();
    protected final boolean hasContainer;
    protected final TaskLifecycleListener.Event lifeCycleEvent;
    protected final String label;
    protected final boolean cancellable;

    public TaskState(boolean hasContainer, TaskLifecycleListener.Event lcEvent, boolean cancellable) {
        this.hasContainer = hasContainer;
        this.lifeCycleEvent = lcEvent;
        this.cancellable = cancellable;
        String name = this.toString();
        name = name.replace("State", "");
        name = name.replaceAll("([a-z]+)([A-Z])", "$1_$2");
        this.label = name.toUpperCase();
    }

    protected void endOrAck(EventContext context) {
        if (context.task.trackingState == Task.TrackingState.START_ACK) {
            this.transition(context, WAIT_END_ACK);
        } else {
            this.taskTerminated(context);
        }
    }

    public void requestContainer(EventContext context) {
        this.illegalState(context, "requestContainer");
    }

    public void containerAllocated(EventContext context, Container container) {
        this.illegalState(context, "containerAllocated");
    }

    public void launchFailed(EventContext context, Throwable t) {
        this.illegalState(context, "launchFailed");
    }

    public void containerStarted(EventContext context) {
        this.illegalState(context, "containerStarted");
    }

    public void startAck(EventContext context) {
        this.illegalState(context, "startAck");
    }

    public void stopTaskFailed(EventContext context, Throwable t) {
        this.illegalState(context, "stopTaskFailed");
    }

    public void completionAck(EventContext context) {
        this.illegalState(context, "completionAck");
    }

    public void containerStopped(EventContext context) {
        this.illegalState(context, "containerStopped");
    }

    public void containerCompleted(EventContext context, ContainerStatus status) {
        this.completed(context, status);
        this.illegalState(context, "containerCompleted");
    }

    public void cancel(EventContext context) {
        this.illegalState(context, "cancel");
    }

    public void tick(EventContext context, long curTime) {
    }

    protected void transition(EventContext context, TaskState newState) {
        TaskState oldState = context.task.state;
        LOG.info((Object)(context.task.getLabel() + " " + oldState.toString() + " --> " + newState.toString()));
        context.task.state = newState;
        if (newState.lifeCycleEvent != oldState.lifeCycleEvent) {
            context.controller.fireLifecycleChange(newState.lifeCycleEvent, context);
        }
        context.task.stateStartTime = System.currentTimeMillis();
    }

    protected void taskStartFailed(EventContext context, Task.Disposition disposition) {
        assert (context.task.container == null);
        context.getTaskManager().completed(context);
        this.taskEnded(context, disposition);
        this.retryTask(context);
    }

    protected void taskTerminated(EventContext context) {
        Task task = context.task;
        context.getTaskManager().completed(context);
        context.group.containerReleased(task);
        assert (task.completionStatus != null);
        if (task.completionStatus.getExitStatus() == 0) {
            this.taskEnded(context, Task.Disposition.COMPLETED);
            context.group.taskEnded(context.task);
        } else {
            this.taskEnded(context, Task.Disposition.RUN_FAILED);
            this.retryTask(context);
        }
    }

    private void taskEnded(EventContext context, Task.Disposition disposition) {
        Task task = context.task;
        if (disposition == null) {
            assert (task.disposition != null);
        } else {
            task.disposition = disposition;
        }
        task.completionTime = System.currentTimeMillis();
        this.transition(context, END);
    }

    private void retryTask(EventContext context) {
        Task task = context.task;
        assert (task.state == END);
        if (!context.controller.isLive() || !task.retryable()) {
            context.group.taskEnded(task);
            return;
        }
        if (task.tryCount > task.taskGroup.getMaxRetries()) {
            LOG.error((Object)(task.getLabel() + " - Too many retries: " + task.tryCount));
            task.disposition = Task.Disposition.TOO_MANY_RETRIES;
            context.group.taskEnded(task);
            return;
        }
        LOG.info((Object)(task.getLabel() + " - Retrying task, try " + task.tryCount));
        context.group.taskRetried(task);
        task.reset();
        this.transition(context, START);
        context.group.enqueuePendingRequest(task);
    }

    private void illegalState(EventContext context, String action) {
        assert (false);
        LOG.error((Object)(context.task.getLabel() + " - Action " + action + " in wrong state: " + this.toString()), (Throwable)new IllegalStateException("Action in wrong state"));
    }

    protected void completed(EventContext context, ContainerStatus status) {
        Task task = context.task;
        String diag = status.getDiagnostics();
        LOG.trace((Object)(task.getLabel() + " Completed, exit status: " + status.getExitStatus() + (String)(DoYUtil.isBlank(diag) ? "" : ": " + status.getDiagnostics())));
        task.completionStatus = status;
    }

    public String toString() {
        return this.getClass().getSimpleName();
    }

    public boolean hasContainer() {
        return this.hasContainer;
    }

    public String getLabel() {
        return this.label;
    }

    public boolean isCancellable() {
        return this.cancellable;
    }

    private static class StartState
    extends TaskState {
        protected StartState() {
            super(false, TaskLifecycleListener.Event.CREATED, true);
        }

        @Override
        public void requestContainer(EventContext context) {
            Task task = context.task;
            ++task.tryCount;
            context.group.dequeuePendingRequest(task);
            if (task.cancelled) {
                this.taskStartFailed(context, Task.Disposition.CANCELLED);
            } else {
                this.transition(context, REQUESTING);
                context.group.enqueueAllocatingTask(task);
                task.containerRequest = context.yarn.requestContainer(task.getContainerSpec());
            }
        }

        @Override
        public void cancel(EventContext context) {
            Task task = context.task;
            assert (!task.cancelled);
            context.group.dequeuePendingRequest(task);
            task.cancel();
            this.taskStartFailed(context, Task.Disposition.CANCELLED);
        }
    }

    private static class RequestingState
    extends TaskState {
        protected RequestingState() {
            super(false, TaskLifecycleListener.Event.CREATED, true);
        }

        @Override
        public void containerAllocated(EventContext context, Container container) {
            Task task = context.task;
            LOG.info((Object)(task.getLabel() + " - Received container: " + DoYUtil.describeContainer(container)));
            context.group.dequeueAllocatingTask(task);
            context.yarn.removeContainerRequest(task.containerRequest);
            task.container = container;
            if (task.cancelled) {
                context.yarn.releaseContainer(container);
                this.taskStartFailed(context, Task.Disposition.CANCELLED);
                return;
            }
            task.error = null;
            task.completionStatus = null;
            this.transition(context, LAUNCHING);
            context.group.containerAllocated(context.task);
            context.getTaskManager().allocated(context);
            try {
                context.yarn.launchContainer(container, task.getLaunchSpec());
                task.launchTime = System.currentTimeMillis();
            }
            catch (YarnFacadeException e) {
                LOG.error((Object)("Container launch failed: " + String.valueOf(task.getContainerId())), (Throwable)e);
                task.error = e;
                context.group.containerReleased(task);
                task.container = null;
                this.taskStartFailed(context, Task.Disposition.LAUNCH_FAILED);
            }
        }

        @Override
        public void cancel(EventContext context) {
            Task task = context.task;
            context.task.cancel();
            LOG.info((Object)(task.getLabel() + " - Cancelled at user request"));
            context.yarn.removeContainerRequest(task.containerRequest);
            context.group.dequeueAllocatingTask(task);
            task.disposition = Task.Disposition.CANCELLED;
            task.completionTime = System.currentTimeMillis();
            this.transition(context, END);
            context.group.taskEnded(context.task);
        }

        @Override
        public void tick(EventContext context, long curTime) {
            Task task = context.task;
            int timeoutSec = task.scheduler.getRequestTimeoutSec();
            if (timeoutSec == 0) {
                return;
            }
            if (task.stateStartTime + (long)(timeoutSec * 1000) > curTime) {
                return;
            }
            LOG.info((Object)(task.getLabel() + " - Request timed out after + " + timeoutSec + " secs."));
            context.yarn.removeContainerRequest(task.containerRequest);
            context.group.dequeueAllocatingTask(task);
            task.disposition = Task.Disposition.LAUNCH_FAILED;
            task.completionTime = System.currentTimeMillis();
            this.transition(context, END);
            context.group.taskEnded(context.task);
            task.scheduler.requestTimedOut();
        }
    }

    private static class LaunchingState
    extends TaskState {
        protected LaunchingState() {
            super(true, TaskLifecycleListener.Event.ALLOCATED, true);
        }

        @Override
        public void launchFailed(EventContext context, Throwable t) {
            Task task = context.task;
            LOG.info((Object)(task.getLabel() + " - Container start failed"));
            context.task.error = t;
            this.launchFailed(context);
        }

        @Override
        public void containerStarted(EventContext context) {
            Task task = context.task;
            if (task.trackingState == Task.TrackingState.NEW) {
                this.transition(context, WAIT_START_ACK);
            } else {
                this.transition(context, RUNNING);
            }
            task.error = null;
            if (task.cancelled) {
                this.transition(context, KILLING);
                context.yarn.killContainer(task.getContainer());
            }
        }

        @Override
        public void startAck(EventContext context) {
            context.task.trackingState = Task.TrackingState.START_ACK;
        }

        @Override
        public void containerCompleted(EventContext context, ContainerStatus status) {
            this.completed(context, status);
            this.endOrAck(context);
        }

        @Override
        public void cancel(EventContext context) {
            context.task.cancel();
            context.yarn.killContainer(context.task.getContainer());
        }

        @Override
        public void tick(EventContext context, long curTime) {
            Task task = context.task;
            if (task.isCancelled() && task.cancellationTime + 10000L < curTime) {
                LOG.error((Object)(task.getLabel() + " - Launch timed out after 10 secs."));
                this.launchFailed(context);
            }
        }

        private void launchFailed(EventContext context) {
            Task task = context.task;
            task.completionTime = System.currentTimeMillis();
            context.yarn.releaseContainer(task.container);
            context.group.containerReleased(task);
            task.container = null;
            this.taskStartFailed(context, Task.Disposition.LAUNCH_FAILED);
        }
    }

    private static class WaitStartAckState
    extends TaskState {
        protected WaitStartAckState() {
            super(true, TaskLifecycleListener.Event.RUNNING, true);
        }

        @Override
        public void startAck(EventContext context) {
            context.task.trackingState = Task.TrackingState.START_ACK;
            this.transition(context, RUNNING);
        }

        @Override
        public void cancel(EventContext context) {
            RUNNING.cancel(context);
        }

        @Override
        public void containerCompleted(EventContext context, ContainerStatus status) {
            this.completed(context, status);
            this.taskTerminated(context);
        }
    }

    private static class RunningState
    extends TaskState {
        protected RunningState() {
            super(true, TaskLifecycleListener.Event.RUNNING, true);
        }

        @Override
        public void containerCompleted(EventContext context, ContainerStatus status) {
            this.completed(context, status);
            this.endOrAck(context);
        }

        @Override
        public void cancel(EventContext context) {
            Task task = context.task;
            task.cancel();
            if (context.group.requestStop(task)) {
                this.transition(context, ENDING);
            } else {
                context.yarn.killContainer(task.container);
                this.transition(context, KILLING);
            }
        }

        @Override
        public void completionAck(EventContext context) {
            context.task.trackingState = Task.TrackingState.END_ACK;
            this.transition(context, ENDING);
        }
    }

    public static class EndingState
    extends TaskState {
        protected EndingState() {
            super(true, TaskLifecycleListener.Event.RUNNING, false);
        }

        @Override
        public void containerCompleted(EventContext context, ContainerStatus status) {
            this.completed(context, status);
            this.endOrAck(context);
        }

        @Override
        public void cancel(EventContext context) {
            context.task.cancel();
        }

        @Override
        public void tick(EventContext context, long curTime) {
            Task task = context.task;
            if (curTime - task.stateStartTime > (long)task.taskGroup.getStopTimeoutMs()) {
                context.yarn.killContainer(task.container);
                this.transition(context, KILLING);
            }
        }

        @Override
        public void completionAck(EventContext context) {
            context.task.trackingState = Task.TrackingState.END_ACK;
        }
    }

    public static class KillingState
    extends TaskState {
        protected KillingState() {
            super(true, TaskLifecycleListener.Event.RUNNING, false);
        }

        @Override
        public void containerCompleted(EventContext context, ContainerStatus status) {
            this.completed(context, status);
            this.endOrAck(context);
        }

        @Override
        public void cancel(EventContext context) {
            context.task.cancel();
        }

        @Override
        public void startAck(EventContext context) {
            context.task.trackingState = Task.TrackingState.START_ACK;
        }

        @Override
        public void completionAck(EventContext context) {
            context.task.trackingState = Task.TrackingState.END_ACK;
        }

        @Override
        public void stopTaskFailed(EventContext context, Throwable t) {
            assert (false);
        }
    }

    private static class WaitEndAckState
    extends TaskState {
        protected WaitEndAckState() {
            super(false, TaskLifecycleListener.Event.RUNNING, false);
        }

        @Override
        public void cancel(EventContext context) {
            context.task.cancel();
        }

        @Override
        public void completionAck(EventContext context) {
            context.task.trackingState = Task.TrackingState.END_ACK;
            this.taskTerminated(context);
        }

        @Override
        public void tick(EventContext context, long curTime) {
            if (!context.getTaskManager().isLive(context)) {
                this.taskTerminated(context);
            }
        }
    }

    private static class EndState
    extends TaskState {
        protected EndState() {
            super(false, TaskLifecycleListener.Event.ENDED, false);
        }

        @Override
        public void cancel(EventContext context) {
        }
    }
}

