/* Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Original Copyright (c) 2005 Covalent Technologies
 *
 * FTP Protocol module for Apache 2.0
 */

#include "mod_ftp.h"
#include "ftp_internal.h"

/* ftp_crlf_filter: Filter for sending ASCII files.  RFC 959 states that
 *                  CRLF should be use where necessary to denote the end
 *                  of line.  This filter replaces all LF's with CRLF's.
 *
 *                  I believe most clients strip the CR's anyway, but we
 *                  need them to transfer the file properly.
 *
 * Arguments: f - The current filter.
 *            bb - The bucket brigade sent down the filter stack.
 *
 * Returns:  APR_SUCCESS on success, otherwise the status code is returned.
 */

apr_status_t ftp_crlf_filter(ap_filter_t *f, apr_bucket_brigade *bb)
{
    apr_bucket *e;
    apr_bucket *b;
    apr_status_t rv;
    apr_off_t this_size = 0;
    int *seen_cr = (int *) f->ctx;

    e = APR_BRIGADE_FIRST(bb);
    while (e != APR_BRIGADE_SENTINEL(bb)) {
        const char *buf;
        apr_size_t len;
        const char *pos;

        if (e->length == 0) {
            e = APR_BUCKET_NEXT(e);     /* onwards */
            continue;
        }

        rv = apr_bucket_read(e, &buf, &len, APR_BLOCK_READ);
        if (rv != APR_SUCCESS) {
            return rv;
        }

        /*
         * Skip all of the extra effort below for empty buckets, if this is
         * EOS we will catch it at the while condition looking for the
         * trailing sentinal.
         */
        if (len == 0) {
            e = APR_BUCKET_NEXT(e);     /* onwards */
            continue;
        }

        /* Handle case where we have a leading LF */
        if (!*seen_cr) {
            if (buf[0] == '\n') {
                b = apr_bucket_immortal_create("\r", 1,
                                               f->c->bucket_alloc);
                APR_BUCKET_INSERT_BEFORE(e, b);
                this_size += 1;
            }
        }
        else {
            *seen_cr = 0;
        }

        /*
         * We search the data for a LF without a preceeding CR. If we find
         * one, we split the bucket so that the LF is the first character in
         * the new bucket, and then insert a new bucket with a CR and insert
         * it before the LF bucket (ignored on the next loop by seen_cr.) As
         * long as we keep seeing CRLF pairs, keep looking forward through
         * the buffer.
         */
        pos = buf;
        while (++pos, pos = memchr(pos, APR_ASCII_LF, len - (pos - buf))) {
            /*
             * We found a bare linefeed, insert a CR and advance to the
             * LF-remainder (we skip that LF above)
             */
            if (*(pos - 1) != APR_ASCII_CR) {
                apr_bucket_split(e, pos - buf);
                e = APR_BUCKET_NEXT(e); /* second half */
                b = apr_bucket_immortal_create("\r", 1,
                                               f->c->bucket_alloc);
                APR_BUCKET_INSERT_BEFORE(e, b);
                this_size += (pos - buf) + 1;   /* including the CR */
                *seen_cr = 1;
                break;
            }
        }

        /*
         * If we just split, we will 'reread' this current bucket... But
         * otherwise we note if the trailing character is a CR, tally this
         * bucket, and advance to the next bucket
         */
        if (!pos) {
            *seen_cr = (buf[len - 1] == '\r');
            this_size += len;
            e = APR_BUCKET_NEXT(e);     /* onwards */
        }

        /*
         * We got too big and chunky, let's spill this out to the client data
         * connection and resume processing
         */
        if (this_size >= 9000) {
            apr_bucket_brigade *bb_split = bb;
            bb = apr_brigade_split(bb_split, e);
            rv = ap_pass_brigade(f->next, bb_split);
            if (rv != APR_SUCCESS)
                return rv;
            this_size = 0;
        }
    }

    if (APR_BRIGADE_EMPTY(bb))
        rv = APR_SUCCESS;
    else {
        rv = ap_pass_brigade(f->next, bb);
        if (rv == APR_SUCCESS && f->c->aborted)
            rv = AP_FILTER_ERROR;
    }
    return rv;
}
