// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

//! CSV formatting support for the row struct objects generated by the library.

use crate::generators::{Building, Customer, Driver, Trip, Vehicle};
use core::fmt;
use std::fmt::Display;

/// Write [`Vehicle`]s in CSV format.
///
/// # Example
/// ```
/// # use spatialbench::generators::VehicleGenerator;
/// # use spatialbench::csv::VehicleCsv;
/// # use std::fmt::Write;
/// // Output the first 3 rows in CSV format
/// let generator = VehicleGenerator::new(1.0, 1, 1);
/// let mut csv = String::new();
/// writeln!(&mut csv, "{}", VehicleCsv::header()).unwrap(); // write header
/// for line in generator.iter().take(3) {
///   // write line using CSV formatter
///   writeln!(&mut csv, "{}", VehicleCsv::new(line)).unwrap();
/// }
/// assert_eq!(
///   csv,
///   "v_vehiclekey,v_mfgr,v_brand,v_type, v_licence\n\
///    1,Manufacturer#1,Brand#13,PROMO BURNISHED COPPER,\"ly. slyly ironi\"\n\
///    2,Manufacturer#1,Brand#13,LARGE BRUSHED BRASS,\"lar accounts amo\"\n\
///    3,Manufacturer#4,Brand#42,STANDARD POLISHED BRASS,\"egular deposits hag\"\n"
/// );
/// ```
pub struct VehicleCsv<'a> {
    inner: Vehicle<'a>,
}

impl<'a> VehicleCsv<'a> {
    pub fn new(inner: Vehicle<'a>) -> Self {
        Self { inner }
    }

    /// Returns the CSV header for the Vehicle table
    pub fn header() -> &'static str {
        "v_vehiclekey,v_mfgr,v_brand,v_type, v_licence"
    }
}

impl Display for VehicleCsv<'_> {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(
            f,
            // note must quote the comment field as it may contain commas
            "{},{},{},{},\"{}\"",
            self.inner.v_vehiclekey,
            self.inner.v_mfgr,
            self.inner.v_brand,
            self.inner.v_type,
            self.inner.v_license
        )
    }
}

/// Write [`Driver`]s in CSV format.
///
/// # Example
/// ```
/// # use spatialbench::generators::DriverGenerator;
/// # use spatialbench::csv::DriverCsv;
/// # use std::fmt::Write;
/// // Output the first 3 rows in CSV format
/// let generator = DriverGenerator::new(1.0, 1, 1);
/// let mut csv = String::new();
/// writeln!(&mut csv, "{}", DriverCsv::header()).unwrap(); // write header
///   for line in generator.iter().take(3) {
///   // write line using CSV formatter
///   writeln!(&mut csv, "{}", DriverCsv::new(line)).unwrap();
/// }
/// assert_eq!(
///   csv,
///   "d_driverkey,d_name,d_address,d_region,d_nation,d_phone\n\
///    1,Driver#000000001,\" N kD4on9OM Ipw3,gf0JBoQDd7tgrzrddZ\",AMERICA,PERU,27-918-335-1736\n\
///    2,Driver#000000002,\"89eJ5ksX3ImxJQBvxObC,\",AFRICA,ETHIOPIA,15-679-861-2259\n\
///    3,Driver#000000003,\"q1,G3Pj6OjIuUYfUoH18BFTKP5aU9bEV3\",AMERICA,ARGENTINA,11-383-516-1199\n"
/// );
/// ```
pub struct DriverCsv {
    inner: Driver,
}

impl DriverCsv {
    pub fn new(inner: Driver) -> Self {
        Self { inner }
    }

    /// Returns the CSV header for the Driver table
    pub fn header() -> &'static str {
        "d_driverkey,d_name,d_address,d_region,d_nation,d_phone"
    }
}

impl Display for DriverCsv {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(
            f,
            // note must quote the address and comment fields as they may contain commas
            "{},{},\"{}\",{},{},{}",
            self.inner.d_driverkey,
            self.inner.d_name,
            self.inner.d_address,
            self.inner.d_region,
            self.inner.d_nation,
            self.inner.d_phone,
        )
    }
}

/// Write [`Customer`]s in CSV format.
///
/// # Example
/// ```
/// # use spatialbench::generators::CustomerGenerator;
/// # use spatialbench::csv::CustomerCsv;
/// # use std::fmt::Write;
/// // Output the first 3 rows in CSV format
/// let generator = CustomerGenerator::new(1.0, 1, 1);
/// let mut csv = String::new();
/// writeln!(&mut csv, "{}", CustomerCsv::header()).unwrap(); // write header
/// for line in generator.iter().take(3) {
///   // write line using CSV formatter
///   writeln!(&mut csv, "{}", CustomerCsv::new(line)).unwrap();
/// }
/// assert_eq!(
///   csv,
///   "c_custkey,c_name,c_address,c_region,c_nation,c_phone\n\
///    1,Customer#000000001,\"IVhzIApeRb ot,c,E\",AFRICA,MOROCCO,25-989-741-2988\n\
///    2,Customer#000000002,\"XSTf4,NCwDVaWNe6tEgvwfmRchLXak\",MIDDLE EAST,JORDAN,23-768-687-3665\n\
///    3,Customer#000000003,\"MG9kdTD2WBHm\",AMERICA,ARGENTINA,11-719-748-3364\n"
/// );
/// ```
pub struct CustomerCsv<'a> {
    inner: Customer<'a>,
}

impl<'a> CustomerCsv<'a> {
    pub fn new(inner: Customer<'a>) -> Self {
        Self { inner }
    }

    /// Returns the CSV header for the Customer table
    pub fn header() -> &'static str {
        "c_custkey,c_name,c_address,c_region,c_nation,c_phone"
    }
}

impl Display for CustomerCsv<'_> {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(
            f,
            // note must quote the address and comment fields as they may contain commas
            "{},{},\"{}\",{},{},{}",
            self.inner.c_custkey,
            self.inner.c_name,
            self.inner.c_address,
            self.inner.c_region,
            self.inner.c_nation,
            self.inner.c_phone,
        )
    }
}

/// Write [`Trip`]s in CSV format.
///
/// # Example
/// ```
/// # use spatialbench::generators::TripGenerator;
/// # use spatialbench::csv::TripCsv;
/// # use std::fmt::Write;
/// // Output the first 3 rows in CSV format
/// let generator = TripGenerator::new(1.0, 1, 1);
/// let mut csv = String::new();
/// writeln!(&mut csv, "{}", TripCsv::header()).unwrap(); // write header
/// for line in generator.iter().take(3) {
///   // write line using CSV formatter
///   writeln!(&mut csv, "{}", TripCsv::new(line)).unwrap();
/// }
/// ```
pub struct TripCsv {
    inner: Trip,
}

impl TripCsv {
    pub fn new(inner: Trip) -> Self {
        Self { inner }
    }

    /// Returns the CSV header for the Trip table
    pub fn header() -> &'static str {
        "t_tripkey,t_custkey,t_driverkey,t_vehiclekey,t_pickuptime,t_dropofftime,t_fare,t_tip,t_totalamount,t_distance,t_pickupx,t_pickupy"
    }
}

impl Display for TripCsv {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(
            f,
            // note must quote location and comment fields as they may contain commas
            "{},{},{},{},{},{},{},{},{},{},\"{:?}\",\"{:?}\"",
            self.inner.t_tripkey,
            self.inner.t_custkey,
            self.inner.t_driverkey,
            self.inner.t_vehiclekey,
            self.inner.t_pickuptime,
            self.inner.t_dropofftime,
            self.inner.t_fare,
            self.inner.t_tip,
            self.inner.t_totalamount,
            self.inner.t_distance,
            self.inner.t_pickuploc,
            self.inner.t_dropoffloc,
        )
    }
}

/// Write [`Building`]s in CSV format.
///
/// # Example
/// ```
/// # use spatialbench::generators::BuildingGenerator;
/// # use spatialbench::csv::BuildingCsv;
/// # use std::fmt::Write;
/// // Output the first 3 rows in CSV format
/// let generator = BuildingGenerator::new(1.0, 1, 1);
/// let mut csv = String::new();
/// writeln!(&mut csv, "{}", BuildingCsv::header()).unwrap(); // write header
/// for line in generator.iter().take(3) {
///   // write line using CSV formatter
///   writeln!(&mut csv, "{}", BuildingCsv::new(line)).unwrap();
/// }
/// ```
pub struct BuildingCsv<'a> {
    inner: Building<'a>,
}

impl<'a> BuildingCsv<'a> {
    pub fn new(inner: Building<'a>) -> Self {
        Self { inner }
    }

    /// Returns the CSV header for the Building table
    pub fn header() -> &'static str {
        "b_buildingkey,b_name,b_boundary"
    }
}

impl Display for BuildingCsv<'_> {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(
            f,
            // note must quote the comment field as it may contain commas
            "{},{},\"{:?}\"",
            self.inner.b_buildingkey, self.inner.b_name, self.inner.b_boundary,
        )
    }
}
