/* 
 *    Programmed By: Mohammed Isam Mohammed [mohammed_isam1984@yahoo.com]
 *    Copyright 2014, 2015, 2016, 2017, 2018 (c)
 * 
 *    file: kbd.c
 *    This file is part of the GnuDOS project.
 *
 *    GnuDOS is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    GnuDOS is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with GnuDOS.  If not, see <http://www.gnu.org/licenses/>.
 */    

#define KBD	1

#include <pthread.h>
#include "kbd.h"

unsigned short mask[] = {192, 224, 240};

static struct termios tty_attr_old;

/*
 * Initialize the terminal, setting the proper flags.
 */
int initTerminal()
{
    struct termios tty_attr;
    tcgetattr(0, &tty_attr_old);
    /* turn off buffering, echo and key processing */
    tty_attr.c_iflag &= ~(IGNBRK | BRKINT | PARMRK | ISTRIP | INLCR | IGNCR | ICRNL | IXON);
    tty_attr.c_oflag &= ~OPOST;
    tty_attr.c_lflag &= ~(ECHO | ECHONL | ICANON | ISIG | IEXTEN);
    tty_attr.c_cflag &= ~(CSIZE | PARENB);
    tty_attr.c_cflag |= CS8;
    tty_attr.c_cc[VMIN] = 1;          /* wait until at least one keystroke available */
    tty_attr.c_cc[VTIME] = 0;         /* no timeout */
    if((tcsetattr(0, TCSAFLUSH, &tty_attr) == -1)) return 0;
    ALT   = 0; 
    CTRL  = 0; 
    SHIFT = 0;
    return 1;
}

/*
 * Restore the terminal to it's previous state.
 * MUST be called before your program exits!.
 */
void restoreTerminal()
{
    tcsetattr(0, TCSAFLUSH, &tty_attr_old);
}

/*
 * SEE: http://www.comptechdoc.org/os/linux/howlinuxworks/linux_hlkeycodes.html
 */
int getKey()
{
    static int bytes = 0;
    static char *sc = uc;
    int c;
    if(bytes)
    {
        c = *sc++;
        bytes--;
        //printf("c = %x [%c], b = %d", c, c, bytes);
        return c;
    }

    bytes = 0;
    sc = uc;
    char *ch = uc;
    int esc = 0;
    c = getchar();
    int extra;
    ALT = 0; CTRL = 0; SHIFT = 0;

    // check for UTF-8 multibyte characters
    if ((c & mask[0]) == mask[0]) bytes++;
    if ((c & mask[1]) == mask[1]) bytes++;
    if ((c & mask[2]) == mask[2]) bytes++;
    if(bytes)
    {
        int b = bytes;
        *ch++ = c;
        while(bytes--)
        {
            c = getchar();
            *ch++ = c;
        }
        bytes = b;
        c = *sc++;
        //printf("!c = %x [%c], b = %d", c, c, bytes);
        return c;
    }

    //printf("c = %x [%c]", c, c);
    // check for possible escape sequences
    if(c == 0x1b)
    {
        esc = 1;
        c = getchar();
    }
    
switchkey:
    
    //printf("c = %x [%c]", c, c);
    switch(c)
    {
        case 0x07:
        case 0x08:
        case 0x09:
        case 0x0a:
        case 0x0b:
        case 0x0c:
        case 0x0d:
        case 0x1b:
            break;
        case 0x5b:
            c = getchar();
            //printf("c = %x [%c]", c, c);
            extra = 1;
            switch(c)
            {
                case 0x31:
                    c = getchar();
                    switch(c)
                    {
                        case 0x3b:
                            CTRL = 1;
                            c = getchar();
                            c = 0x5b;
                            goto switchkey;
                        case 0x7e: c = HOME_KEY; break;
                        /*
                         * TODO: handle function keys here. See the link above.
                         */
                        default:   c = '['; break;
                    }
                    break;
                case 0x32:
                    c = getchar();
                    switch(c)
                    {
                        case 0x7e: c = INS_KEY; break;
                        /* ditto ... */
                        default:   c = '['; break;
                    }
                    break;
                case 0x33:
                    c = getchar();
                    //printf("c = %x [%c]", c, c);
                    switch(c)
                    {
                        case 0x3b:
                            CTRL = 1;
                            c = getchar();
                            goto switchkey;
                        case 0x7e: c = DEL_KEY; break;
                        /* ditto ... */
                        default:   c = '['; break;
                    }
                    extra = 0;
                    break;
                case 0x34:
                    c = END_KEY;
                    break;
                case 0x35:
                    c = PGUP_KEY;
                    break;
                case 0x36:
                    c = PGDOWN_KEY;
                    break;
                case 0x41:
                    c = UP_KEY;
                    extra = 0;
                    break;
                case 0x42:
                    c = DOWN_KEY;
                    extra = 0;
                    break;
                case 0x43:
                    c = RIGHT_KEY;
                    extra = 0;
                    break;
                case 0x44:
                    c = LEFT_KEY;
                    extra = 0;
                    break;
                case 0x46:
                    c = END_KEY;
                    extra = 0;
                    break;
                case 0x47:
                    c = '5';
                    extra = 0;
                    break;
                case 0x48:
                    c = HOME_KEY;
                    extra = 0;
                    break;
                case 0x50:
                        /*
                         * TODO: this should be the Pause key.
                         */
                    c = '[';
                    extra = 0;
                    break;
                case 0x5b:
                    c = getchar();
                    /*
                     * TODO: handle function keys here. See the link above.
                     */
                    switch(c)
                    {
                        case 0x33: c = DEL_KEY; break;
                        /* ditto ... */
                        default:   c = '['; break;
                    }
                    break;
                default:
                    //printf("c = %x [%c]", c, c);
                    break;
            }
            if(extra)
            {
                while(getchar() != 0x7e) ;
            }
            esc = 0;
            break;
        case 0x7f:
            c = '\b';
            break;
        default:
            //printf("c = %x [%c]", c, c);
            if(c < 0x20)
            {
                CTRL = 1;
                c += 0x40;
            }
            //printf("c = %x [%c]", c, c);
            break;
    }

    *ch++ = c;
    *ch   = '\0';
    if(esc) ALT = 1;
    //printf("c = %x [%c]", c, c);
    return c;
}
