#!/usr/bin/env bash

# This script generates tests text-emphasis-style-property-010* except
# 010Cn. The tests generated cover all characters listed in the unicode
# data file which should not have emphasis mark specified in the spec.
# This script downloads UnicodeData.txt from the website of the Unicode
# Consortium and extract the characters form that file. It requires
# python (either 2.5+ or 3.x), awk, and wget to work. Only test files
# are generated by this script. It also outputs a list of all tests it
# generated in the format of GNU reftest.list to the stdout. Other
# information has been redirected to the stderr.

UNICODE_DATA_FILE='UnicodeData.txt'
UNICODE_DATA_URL="http://www.unicode.org/Public/8.0.0/ucd/$UNICODE_DATA_FILE"
UNICODE_DATA_DIGEST='38b17e1118206489a7e0ab5d29d7932212d38838df7d3ec025ecb58e8798ec20'

TEST_FILE='text-emphasis-style-property-010%s.html'
REF_FILE='text-emphasis-style-property-010-ref.html'

digest_file() {
    python -c "import hashlib;
print(hashlib.sha256(open('$1', 'rb').read()).hexdigest())"
}

check_file() {
    [[ -f "$UNICODE_DATA_FILE" ]] || return 1
    digest=`digest_file "$UNICODE_DATA_FILE"`
    [[ "$digest" == "$UNICODE_DATA_DIGEST" ]] || return 2
}

download_data() {
    check_file
    if [[ $? -eq 2 ]]; then
        echo "Removing incorrect data file..." >&2
        rm "$UNICODE_DATA_FILE"
    fi
    wget -nc -O"$UNICODE_DATA_FILE" "$UNICODE_DATA_URL" >&2

    check_file
    if [[ $? -ne 0 ]]; then
        echo "Failed to get the correct unicode data file!" >&2
        exit 1
    fi
}

list_codepoints() {
    awk -F';' "\$3 == \"$1\" { print \"    0x\"\$1\",\" }" "$UNICODE_DATA_FILE"
}

write_test_file() {
    filename=`printf "$TEST_FILE" $1`
    echo "== $filename $REF_FILE"
    cat <<EOF > $filename
<!DOCTYPE html>
<meta charset="utf-8">
<title>CSS Test: text-emphasis, $1</title>
<link rel="author" title="Xidorn Quan" href="https://www.upsuper.org">
<link rel="author" title="Mozilla" href="https://www.mozilla.org">
<link rel="help" href="https://drafts.csswg.org/css-text-decor-3/#text-emphasis-style-property">
<meta name="assert" content="Emphasis marks should not be rendered for characters in general category $1">
<link rel="match" href="text-emphasis-style-property-010-ref.html">
<p>Pass if there is nothing rendered below:</p>
<div style="color: white; white-space: pre-wrap; text-emphasis: filled circle red">
<script>
  var codepoints = [
`list_codepoints "$1"`
  ];
  document.write(codepoints.map(function (code) {
    return String.fromCodePoint(code);
  }).join(' '));
</script>
</div>
EOF
}

download_data
echo "# START tests from $0"
for c in Zs Zl Zp Cc Cf; do
    write_test_file "$c"
done
echo "# END tests from $0"
