/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { Configuration, j2s } from "@gnu-taler/taler-util";
import {
  configureCommonKyc,
  createKycTestkudosEnvironment,
  makeTestPaymentV2,
  withdrawViaBankV3,
} from "../harness/environments.js";
import {
  doMerchantKycAuth,
  GlobalTestState,
  harnessHttpLib,
} from "../harness/harness.js";

function adjustExchangeConfig(config: Configuration) {
  configureCommonKyc(config);

  config.setString("KYC-RULE-R1", "operation_type", "aggregate");
  config.setString("KYC-RULE-R1", "enabled", "yes");
  config.setString("KYC-RULE-R1", "exposed", "yes");
  config.setString("KYC-RULE-R1", "is_and_combinator", "yes");
  config.setString("KYC-RULE-R1", "threshold", "TESTKUDOS:5");
  config.setString("KYC-RULE-R1", "timeframe", "1d");
  config.setString("KYC-RULE-R1", "next_measures", "M1");

  config.setString("KYC-MEASURE-M1", "check_name", "C1");
  config.setString("KYC-MEASURE-M1", "context", "{}");
  config.setString("KYC-MEASURE-M1", "program", "NONE");

  config.setString("KYC-CHECK-C1", "type", "INFO");
  config.setString("KYC-CHECK-C1", "description", "my check!");
  config.setString("KYC-CHECK-C1", "fallback", "FREEZE");
}

export async function runKycMerchantAggregateTest(t: GlobalTestState) {
  // Set up test environment

  const {
    merchant,
    walletClient,
    bankClient,
    exchange,
    bank,
    exchangeBankAccount, merchantAdminAccessToken,
  } = await createKycTestkudosEnvironment(t, { adjustExchangeConfig });

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    bankClient,
    amount: "TESTKUDOS:50",
    exchange: exchange,
    walletClient: walletClient,
  });

  await wres.withdrawalFinishedCond;

  await doMerchantKycAuth(t, {
    bankClient,
    exchangeBankAccount,
    merchantAdminAccessToken,
    merchant,
    bankAdminAuth: bank.getAdminAuth(),
  });


  await makeTestPaymentV2(t, {
    merchant, merchantAdminAccessToken,
    walletClient,
    order: {
      amount: "TESTKUDOS:20",
      summary: "hello",
    },
  });

  await exchange.runAggregatorOnceWithTimetravel({
    timetravelMicroseconds: 1000 * 1000 * 60 * 60,
  });

  await exchange.runTransferOnceWithTimetravel({
    timetravelMicroseconds: 1000 * 1000 * 60 * 60,
  });

  t.logStep("start-run-kyccheck");

  await merchant.runReconciliationOnceWithTimetravel({
    timetravelMicroseconds: 1000 * 1000 * 60 * 60,
  });

  t.logStep("start-request-kyc");
  const kycStatusUrl = new URL("private/kyc", merchant.makeInstanceBaseUrl());
  const resp = await harnessHttpLib.fetch(kycStatusUrl.href, {
    headers: {
      Authorization: `Bearer ${merchantAdminAccessToken}`
    }
  });

  console.log(`mechant kyc status: ${resp.status}`);

  t.assertDeepEqual(resp.status, 200);

  console.log(j2s(await resp.json()));
}

runKycMerchantAggregateTest.suites = ["wallet"];
