/*  ADCD - A Diminutive CD player for GNU/Linux
    Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <cstdio>
#include <cstdlib>
#include <vector>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>

#include "msf_time.h"
#include "cd.h"


CD::CD( const char * const filename )
  : status( no_disc ),
    _track( 0 ), _first_track( 0 ), _last_track( 0 ), _index( 0 ), _loop( 0 ),
    _linear( true )
  {
  filedes = ::open( filename, O_RDONLY | O_NONBLOCK );
  if( filedes < 0 )
    {
    std::fprintf( stderr, "adcd: cannot open '%s' in non-blocking mode.\n", filename );
    std::fprintf( stderr, "Please, verify if you have permission to read device '%s'.\n", filename );
    exit( 1 );
    }
  read_status();
  }


CD::~CD() { if ( filedes >= 0 ) ::close( filedes ); }


bool CD::read_status( const bool force )
  {
  if( !force && ( status == no_disc || status == stopped ) ) return false;
  const Status old_status = status;
  const int old_tracks = tracks();
  switch( ioctl( filedes, CDROM_DRIVE_STATUS ) )
    {
    case CDS_NO_INFO:
    case CDS_DISC_OK:
      if( status == no_disc || status == stopped )
        {
        struct cdrom_tochdr tinfo;
        if( ioctl( filedes, CDROMREADTOCHDR, &tinfo ) == 0 )
          {
          _first_track = tinfo.cdth_trk0; _last_track = tinfo.cdth_trk1;
          if( tracks() )
            {
            _timelist.resize( tracks() );
            int i;
            for( i = 0; i <= tracks(); ++i )
              {
              struct cdrom_tocentry entry;
              entry.cdte_track = (i < tracks()) ? i+_first_track : CDROM_LEADOUT;
              entry.cdte_format = CDROM_MSF;
              if( ioctl( filedes, CDROMREADTOCENTRY, &entry ) != 0 ) break;
              int m = entry.cdte_addr.msf.minute;
              int s = entry.cdte_addr.msf.second;
              int f = entry.cdte_addr.msf.frame;
              if( i > 0 ) _timelist[i-1].end = Msf_time( m, s, f - 1 );
              if( i < tracks() ) _timelist[i].start = Msf_time( m, s, f );
              }
            if( i > tracks() )
              {
              if( _track < _first_track ) _track = _first_track;
              else if( _track > _last_track ) _track = _last_track;
              _time_abs = time_start( _track ); _time_rel = 0;
              status = stopped;
              }
            }
          }
        }
      break;
    case CDS_NO_DISC:
    case CDS_TRAY_OPEN:
    case CDS_DRIVE_NOT_READY:
    default                 : status = no_disc;
    }
  if( status != no_disc )
    {
    struct cdrom_subchnl ch;
    ch.cdsc_format = CDROM_MSF;
    if( ioctl( filedes, CDROMSUBCHNL, &ch ) != 0 ) status = no_disc;
    else switch( ch.cdsc_audiostatus )
      {
      case CDROM_AUDIO_PLAY:
        status = playing;
        if( _loop != 2 ) _track = ch.cdsc_trk;
        _time_abs = Msf_time( ch.cdsc_absaddr.msf.minute,
                              ch.cdsc_absaddr.msf.second,
                              ch.cdsc_absaddr.msf.frame );
        _time_rel = Msf_time( ch.cdsc_reladdr.msf.minute,
                              ch.cdsc_reladdr.msf.second,
                              ch.cdsc_reladdr.msf.frame );
        break;
      case CDROM_AUDIO_PAUSED   : status = paused; break;
      case CDROM_AUDIO_INVALID  :
      case CDROM_AUDIO_NO_STATUS:
      case CDROM_AUDIO_COMPLETED:
        if( status == playing )
          {
          if( _loop == 2 || ( _linear && _time_rel <= Msf_time( 0, 2 ) ) ) play();
          else if( !next_track() ) status = stopped;
          }
        break;
      case CDROM_AUDIO_ERROR: status = stopped; break;
      }
    }
  if( status == no_disc && old_status != no_disc )
    {
    _track = _first_track = _last_track = 0;
    _time_abs = _time_rel = 0; _timelist.clear();
    }
  if( tracks() != old_tracks )
    { _index = 0; _linear = true; _playlist.clear(); }
  return ( status != no_disc || status != old_status || tracks() != old_tracks );
  }


void CD::close()
  {
  if( status == no_disc ) { ioctl( filedes, CDROMCLOSETRAY ); read_status(); }
  }


void CD::open()
  {
  stop(); ioctl( filedes, CDROMEJECT ); read_status();
  }


void CD::pause()
  {
  if( status == playing ) { ioctl( filedes, CDROMPAUSE ); status = paused; }
  else if( status == paused )
    { ioctl( filedes, CDROMRESUME ); status = playing; read_status(); }
  }


void CD::play()
  {
  if( status == paused ) { pause(); return; }
  if( status == stopped ) read_status();
  if( status == no_disc ) { close(); if( status == no_disc ) return; }

  struct cdrom_ti ti;
  ti.cdti_ind0 = ti.cdti_ind1 = 1;			// FIXME Linus
  if( _loop == 2 ) ti.cdti_trk0 = ti.cdti_trk1 = _track;
  else if( _linear ) { ti.cdti_trk0 = _track; ti.cdti_trk1 = _last_track; }
  else if( _playlist.size() )
    {
    if( _index >= (int)_playlist.size() ) _index = _playlist.size() - 1;
    ti.cdti_trk0 = ti.cdti_trk1 = _track = _playlist[_index];
    }
  else return;
  if( !tracks() || _track < _first_track || _track > _last_track ) return;
  ioctl( filedes, CDROMPLAYTRKIND, &ti );
  read_status();
  }


void CD::stop()
  {
  if( status == playing || status == paused )
    { ioctl( filedes, CDROMSTOP ); status = stopped; }
  }


bool CD::next_track()
  {
  if( _linear )
    {
    if( _track >= _first_track - 1 && _track < _last_track ) ++_track;
    else if( _loop ) _track = _first_track;
    else return false;
    }
  else if( _playlist.size() )
    {
    if( _index < (int)_playlist.size() - 1 ) _track = _playlist[++_index];
    else if( _loop ) { _index = 0; _track = _playlist[_index]; }
    else return false;
    }
  else return false;
  return track( _track );
  }


bool CD::prev_track()
  {
  if( _linear )
    {
    if( _track > _first_track && _track <= _last_track + 1 ) --_track;
    else if( _loop ) _track = _last_track;
    else return false;
    }
  else if( _playlist.size() )
    {
    if( _index > 0 ) _track = _playlist[--_index];
    else if( _loop )
      { _index = _playlist.size() - 1; _track = _playlist[_index]; }
    else return false;
    }
  else return false;
  return track( _track );
  }


bool CD::seek_forward( const int seconds )
  {
  if( status != playing || !_timelist.size() ||
      ( !_linear && !_playlist.size() ) ) return false;

  Msf_time target = _time_abs + Msf_time( 0, seconds );
  Msf_time end = _timelist.back().end;
  if( !_linear || _loop == 2 ) end = time_end( _track );
  if( target >= end )
    {
    if( ( !_linear || _loop ) && next_track() ) return true;
    else target = end - Msf_time( 0, 0, 1 );
    }
  struct cdrom_msf msf;
  msf.cdmsf_min0 = target.minute();
  msf.cdmsf_sec0 = target.second();
  msf.cdmsf_frame0 = target.frame();
  msf.cdmsf_min1 = end.minute();
  msf.cdmsf_sec1 = end.second();
  msf.cdmsf_frame1 = end.frame();
  ioctl( filedes, CDROMPLAYMSF, &msf );
  read_status();
  return true;
  }


bool CD::seek_backward( const int seconds )
  {
  if( status != playing || !_timelist.size() ||
      ( !_linear && !_playlist.size() ) ) return false;

  Msf_time target = _time_abs - Msf_time( 0, seconds );
  Msf_time start = _timelist.front().start;
  Msf_time end = _timelist.back().end;
  if( !_linear || _loop == 2 )
    { start = time_start( _track ); end = time_end( _track ); }
  if( target < start )
    {
    if( ( !_linear || _loop ) && prev_track() ) return true;
    if( _linear ) _track = _first_track;
    _time_abs = start; _time_rel = 0; stop(); return true;
    }
  struct cdrom_msf msf;
  msf.cdmsf_min0 = target.minute();
  msf.cdmsf_sec0 = target.second();
  msf.cdmsf_frame0 = target.frame();
  msf.cdmsf_min1 = end.minute();
  msf.cdmsf_sec1 = end.second();
  msf.cdmsf_frame1 = end.frame();
  ioctl( filedes, CDROMPLAYMSF, &msf );
  read_status();
  return true;
  }


void CD::loop( const int new_loop )
  {
  _loop = (new_loop < 0) ? 0 : new_loop % 3;
  if( _loop == 2 && _linear && status == playing ) seek_forward( 1 );
  }


const char * CD::loop_name() const throw()
  {
  switch( _loop )
    {
    case 0 : return "No   ";
    case 1 : return "Disc ";
    case 2 : return "Track";
    default: return "error";
    }
  }


void CD::playlist( const std::vector< int > & pl )
  {
  _playlist = pl;
  if( pl.size() == 0 ) _index = 0;
  else if( _index >= (int)pl.size() ) _index = pl.size() - 1;
  }


void CD::show_info() const throw()
  {
  switch( status )
    {
    case no_disc: std::printf( "No Disc\n" ); break;
    case stopped: std::printf( "Stopped\n" ); break;
    case paused : std::printf( "Paused\n" ); break;
    case playing:
      {
      Msf_time msf = time( relative ), total = time_track( _track );
      std::printf( "Playing track %d / %d -- time %d:%02d / %d:%02d -- volume %d\n",
                   _track, _last_track, msf.minute(), msf.second(),
                   total.minute(), total.second(), volume() );
      } break;
    default:      std::printf( "Unknown\n" );
    }
  }


const char * CD::status_name() const throw()
  {
  switch( status )
    {
    case no_disc: return "No Disc";
    case stopped: return "Stopped";
    case paused : return "Paused ";
    case playing: return "Playing";
    default:      return "Unknown";
    }
  }


bool CD::track( const int new_track, const bool start )
  {
  if( start )
    {
    if( status == stopped ) read_status();
    if( status == no_disc ) { close(); if( status == no_disc ) return false; }
    }
  if( !tracks() || new_track < _first_track || new_track > _last_track )
    return false;

  _track = new_track;
  if( !_linear && _playlist.size() && _playlist[_index] != _track )
    {
    for( unsigned int i = _index + 1; i < _playlist.size(); ++i )
      if( _track == _playlist[i] ) { _index = i; break; }
    if( _playlist[_index] != _track )
      for( unsigned int i = 0; i < _playlist.size(); ++i )
        if( _track == _playlist[i] ) { _index = i; break; }
    if( _playlist[_index] != _track ) _linear = true;
    }
  if( status == paused || start ) status = playing;
  if( status == playing ) play();
  else { _time_abs = time_start( _track ); _time_rel = 0; }
  return true;
  }


int CD::volume() const throw()
  {
  struct cdrom_volctrl volctrl;
  if( ioctl( filedes, CDROMVOLREAD, &volctrl ) == 0 )
    return ( volctrl.channel0 + volctrl.channel1 ) / 2;
  else return 0;
  }


bool CD::volume( int vol ) const throw()
  {
  if( vol < 0 ) vol = 0; else if( vol > 255 ) vol = 255;
  struct cdrom_volctrl volctrl;
  volctrl.channel0 = volctrl.channel1 = vol;
  return ( ioctl( filedes, CDROMVOLCTRL, &volctrl ) == 0 );
  }


Msf_time CD::time( const Time_mode mode ) const throw()
  {
  if( _timelist.size() ) switch( mode )
    {
    case relative: return _time_rel;
    case rem_rel : return time_end( _track ) - time_start( _track ) - _time_rel;
    case absolute: return _time_abs;
    case rem_abs : return _timelist.back().end - _time_abs;
    }
  return 0;
  }


Msf_time CD::time_start( const int track ) const throw()
  {
  if( _timelist.size() && tracks() &&
      track >= _first_track && track <= _last_track )
    return _timelist[track-_first_track].start;
  return 0;
  }


Msf_time CD::time_end( const int track ) const throw()
  {
  if( _timelist.size() && tracks() &&
      track >= _first_track && track <= _last_track )
    return _timelist[track-_first_track].end;
  return 0;
  }


Msf_time CD::time_track( const int track ) const throw()
  {
  if( _timelist.size() && tracks() &&
      track >= _first_track && track <= _last_track )
    return _timelist[track-_first_track].end - _timelist[track-_first_track].start;
  return 0;
  }


Msf_time CD::time_disc() const throw()
  {
  if( _timelist.size() )
    return _timelist.back().end - _timelist.front().start;
  return 0;
  }
