/*  ADCD - A Diminutive CD player for GNU/Linux
    Copyright (C) 2004, 2005, 2006, 2007, 2009, 2010 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/*
    Return values: 0 for a normal exit, 1 for environmental problems
    (file not found, invalid flags, I/O errors, etc), 2 to indicate a
    corrupt or invalid input file, 3 for an internal consistency error
    (eg, bug) which caused adcd to panic.
*/

#include <cctype>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <string>
#include <vector>

#include "arg_parser.h"
#include "msf_time.h"
#include "cd.h"
#include "player.h"


namespace {

const char * invocation_name = 0;
const char * const Program_name    = "Adcd";
const char * const program_name    = "adcd";
const char * const program_year    = "2010";


void show_error( const char * msg, const int errcode = 0, const bool help = false ) throw()
  {
  if( msg && msg[0] != 0 )
    {
    std::fprintf( stderr, "%s: %s", program_name, msg );
    if( errcode > 0 ) std::fprintf( stderr, ": %s", strerror( errcode ) );
    std::fprintf( stderr, "\n" );
    }
  if( help && invocation_name && invocation_name[0] != 0 )
    std::fprintf( stderr, "Try `%s --help' for more information.\n", invocation_name );
  }


void show_help() throw()
  {
  std::printf( "%s - A Diminutive CD player for GNU/Linux.\n", Program_name );
  std::printf( "If invoked without options enters interactive mode.\n" );
  std::printf( "\nUsage: %s [options] [device]\n", invocation_name );
  std::printf( "\nOptions:\n" );
  std::printf( "  -h, --help             display this help and exit\n" );
  std::printf( "  -V, --version          output version information and exit\n" );
  std::printf( "  -i, --info             display info about current status and exit\n" );
  std::printf( "  -p, --play[=<n>]       start playing at track <n> (default 1) and exit\n" );
  std::printf( "  -s, --stop             stop playing and exit\n" );
  std::printf( "  -u, --pause            pause/resume playing and exit\n" );
  std::printf( "  -v, --volume[=[+-]<n>] show, adjust or set volume and exit\n" );
  std::printf( "\nKeys not shown on screen:\n" );
  std::printf( "  1 to 9             Start playing at track 1 to 9\n" );
  std::printf( "  0                  Start playing at track 10\n" );
  std::printf( "  F1 to F12          Start playing at track 11 to 22\n" );
  std::printf( "  up arrow           Seek backward 10 seconds\n" );
  std::printf( "  down arrow         Seek forward 10 seconds\n" );
  std::printf( "  Page up            Seek backward 30 seconds\n" );
  std::printf( "  Page down          Seek forward 30 seconds\n" );
  std::printf( "  Home               Seek backward 1 minute\n" );
  std::printf( "  End                Seek forward 1 minute\n" );
  std::printf( "  Ins                Seek backward 5 minutes\n" );
  std::printf( "  Del                Seek forward 5 minutes\n" );
  std::printf( "  + -                increase/decrease volume by  1\n" );
  std::printf( "  * /                increase/decrease volume by 10\n" );
  std::printf( "  L                  Loop mode (no_loop, Disc, Track)\n" );
  std::printf( "  M                  Change playing mode (Linear/Playlist)\n" );
  std::printf( "  R                  Generate random playlist\n" );
  std::printf( "  T                  Change displayed time\n" );
  std::printf( "\nKeys used for editing playlist:\n" );
  std::printf( "  arrow keys         Move the cursor\n" );
  std::printf( "  0..9, F1..F12      Insert corresponding track\n" );
  std::printf( "  + -                Increase/decrease track by  1\n" );
  std::printf( "  * /                Increase/decrease track by 10\n" );
  std::printf( "  Del                Delete track at cursor\n" );
  std::printf( "  Backspace          Delete track preceding cursor\n" );
  std::printf( "  Enter              Store playlist and exit edit mode\n" );
  std::printf( "  Q                  Discard changes and exit edit mode\n" );
  std::printf( "\nReport bugs to adcd-bug@nongnu.org\n" );
  std::printf( "Adcd home page: http://www.nongnu.org/adcd/adcd.html\n" );
  }


void show_version() throw()
  {
  std::printf( "%s %s\n", Program_name, PROGVERSION );
  std::printf( "Copyright (C) %s Antonio Diaz Diaz.\n", program_year );
  std::printf( "License GPLv3+: GNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>\n" );
  std::printf( "This is free software: you are free to change and redistribute it.\n" );
  std::printf( "There is NO WARRANTY, to the extent permitted by law.\n" );
  }

} // end namespace


int main( const int argc, const char * argv[] )
  {
//  const char * mixer_dev_name = "/dev/mixer";
  enum Volcommand { vnone, vadjust, vset, vshow };
  Volcommand vc = vnone;
  int track = 1, volume = 0;
  bool info = false, pause = false, play = false, stop = false;
  invocation_name = argv[0];

  const Arg_parser::Option options[] =
    {
    { 'h', "help",    Arg_parser::no    },
    { 'i', "info",    Arg_parser::no    },
//    { 'm', "mixer",   Arg_parser::yes   },
    { 'p', "play",    Arg_parser::maybe },
    { 's', "stop",    Arg_parser::no    },
    { 'u', "pause",   Arg_parser::no    },
    { 'v', "volume",  Arg_parser::maybe },
    { 'V', "version", Arg_parser::no    },
    {  0 , 0,         Arg_parser::no    } };

  Arg_parser parser( argc, argv, options );
  if( parser.error().size() )				// bad option
    { show_error( parser.error().c_str(), 0, true ); return 1; }

  int argind = 0;
  for( ; argind < parser.arguments(); ++argind )
    {
    const int code = parser.code( argind );
    if( !code ) break;					// no more options
    const char * arg = parser.argument( argind ).c_str();
    switch( code )
      {
      case 'h': show_help(); return 0;
      case 'i': info = true; break;
//      case 'm': mixer_dev_name = arg; break;
      case 'p': play = true; if( arg[0] ) track = std::strtol( arg, 0, 0 ); break;
      case 's': stop = true; break;
      case 'u': pause = true; break;
      case 'v': if( !arg[0] ) vc = vshow;
                else if( isdigit( arg[0] ) ) vc = vset;
                else if( ( arg[0] == '+' || arg[0] == '-' ) && isdigit( arg[1] ) )
                  vc = vadjust;
                else { show_error( "bad volume value" ); return 1; }
                volume = std::strtol( arg, 0, 0 );
                break;
      case 'V':	show_version(); return 0;
      default : show_error( "internal error: uncaught option" ); return 3;
      }
    } // end process options

  const char * cd_dev_name = "/dev/cdrom";
  if( argind < parser.arguments() ) cd_dev_name = parser.argument( argind++ ).c_str();
  if( argind < parser.arguments() )
    { show_error( "too many devices", 0, true ); return 1; }

  // end scan arguments

  CD cd( cd_dev_name );
  bool interactive = true;

  if( vc != vnone )
    {
    interactive = false;
    if( vc == vshow ) std::printf( "Current volume = %d\n", cd.volume() );
    else
      {
      if( vc == vadjust ) cd.volume( cd.volume() + volume );
      else if( vc == vset ) cd.volume( volume );
      std::printf( "New volume = %d\n", cd.volume() );
      }
    }

  if( stop ) { cd.stop(); interactive = false; }
  if( play )
    {
    if( !cd.track( track, true ) ) { show_error( "bad track number" ); return 1; }
    interactive = false;
    }
  if( info ) { cd.show_info(); interactive = false; }
  if( pause ) { cd.pause(); interactive = false; if( info ) cd.show_info(); }

  if( interactive ) Player::main_loop( cd );
  return 0;
  }
