%module base
%feature("autodoc", "1");   // enable docstrings for python wrappers

%{
#include <string>
#include "base/nls.h"
#include "base/base.h"
#include "base/endians.h"
#include "base/types.h"
#include "base/diskio.h"
#include "base/configuration.h"
%}

%include "stdint.i"
%include "std_string.i"

// typemaps for the flat::next (...) method
namespace base {
    %typemap(in, numinputs = 0) (void **value, u_int32 *size, char **name) "
        void *value;
        char *name;
        u_int32 size;
        
        $1 = &value;
        $2 = &size;
        $3 = &name;
    "
    %typemap(argout) (void **value, u_int32 *size, char **name) {
        Py_XDECREF($result);
        PyObject *py_value;
        switch (result) {
            case base::flat::T_BOOL:
            case base::flat::T_UINT8:
            {
                py_value = PyInt_FromLong (*((u_int8*) *$1));
                break;
            }
            case base::flat::T_SINT8:
            {
                py_value = PyInt_FromLong (*((s_int8*) *$1));
                break;
            }
            case base::flat::T_SINT16:
            {
                py_value = PyInt_FromLong (*((s_int16*) *$1));
                break;
            }
            case base::flat::T_UINT16:
            {
                py_value = PyInt_FromLong (*((u_int16*) *$1));
                break;
            }
            case base::flat::T_SINT32:
            {
                py_value = PyLong_FromLong (*((s_int32*) *$1));
                break;
            }
            case base::flat::T_UINT32:
            {
                py_value = PyLong_FromUnsignedLong (*((u_int32*) *$1));
                break;
            }
            case base::flat::T_FLOAT:
            case base::flat::T_DOUBLE:
            {
                py_value = PyFloat_FromDouble (atof ((char *) *$1));
                break;
            }
            case base::flat::T_CHAR:
            case base::flat::T_STRING:
            case base::flat::T_BLOB:
            {
                py_value = PyString_FromStringAndSize ((char *) *$1, *$2);
                break;
            }
            case base::flat::T_FLAT:
            {
                base::flat *flat = new base::flat ((const char *) *$1, *$2);
                py_value = SWIG_NewPointerObj ((void *) flat, $descriptor(base::flat*), 1);
                break;
            }
            default:
            {
                return Py_BuildValue ("(iOOO)", result, Py_None, Py_None, Py_None);
            }
        }
        
        $result = Py_BuildValue ("(iOis)", result, py_value, *$2, (char *) *$3);
    }

    // typemap to let C++ have ownership of cfg_option* given to configuration::add_option ()
    %typemap(in) cfg_option *value "if ((SWIG_ConvertPtr ($input, (void **) &$1, $1_descriptor, SWIG_POINTER_EXCEPTION | SWIG_POINTER_DISOWN)) == -1) SWIG_fail;"
}

// typemap for passing a FILE* from Python
%typemap(in) FILE * { 
    if (!PyFile_Check ($input)) { 
        PyErr_SetString(PyExc_TypeError, "Need a file!"); 
        goto fail;
    } 
    $1 = PyFile_AsFile($input); 
} 


// typemap for returning a vector<const char*> as python list
%typemap(out) vector<const char*> {
    unsigned int index = 0;
    $result = PyList_New (result.size ());
    for (vector<const char*>::iterator i = result.begin (); i != result.end (); i++)
        PyList_SET_ITEM ($result, index++, PyString_FromString (*i));
}

%include "base/base.h"
%include "base/types.h"
%include "base/timer.h"
%include "base/file.h"
%include "base/flat.h"
%include "base/diskio.h"
%include "base/configuration.h"
%include "base/paths.h"

/* implement friend operators of igzstream */
%extend base::igzstream {
    /// Reads a boolean.
    bool get_bool ()
    {
        u_int8 b;
        b << *self;
        return b;
    }
        
    /// Reads a u_int8.
    u_int8 get_uint8 ()
    {
        u_int8 n;
        n << *self;
        return n;
    }
    
    /// Reads a s_int8.
    s_int8 get_sint8 ()
    {
        s_int8 n;
        n << *self;
        return n;
    }
    
    /// Reads a u_int16.
    u_int16 get_uint16 ()
    {
        u_int16 n;
        n << *self;
        return n;
    }
    
    /// Reads a s_int16.
    s_int16 get_sint16 ()
    {
        s_int16 n;
        n << *self;
        return n;
    }
    
    /// Reads a u_int32.
    u_int32 get_uint32 ()
    {
        u_int32 n;
        n << *self;
        return n;
    }
    
    /// Reads a s_int32.
    s_int32 get_sint32 ()
    {
        s_int32 n;
        n << *self;
        return n;
    }
    
    /// Reads a string.
    string get_string ()
    {
        string s;
        s << *self;
        return s;
    }
    
    /// Reads a float.
    float get_float ()
    {
        float f;
        f << *self;
        return f;
    }
}


/* implement friend operators of ogzstream */
%extend base::ogzstream 
{
    void put_bool (const bool &b)         { b >> *self; }
    void put_uint8 (const u_int8 &n)      { n >> *self; }
    void put_sint8 (const s_int8 &n)      { n >> *self; }
    void put_uint16 (const u_int16 &n)    { n >> *self; }
    void put_sint16 (const s_int16 &n)    { n >> *self; }
    void put_uint32 (const u_int32 &n)    { n >> *self; }
    void put_sint32 (const s_int32 &n)    { n >> *self; }
    void put_string (const string& s)     { s >> *self; }
    void put_float (const float &f)       { f >> *self; }
}

%pythoncode %{
    Timer = cvar.Timer
    Paths = cvar.Paths
%}
