/*
 *   This file is part of AkariXB
 *   Copyright 2015-2016  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "commandmodule.h"


CommandModule::CommandModule(CommandHandler *handler,
                             GlobalObject *globalObject,
                             QWidget *parent) : QWidget(parent)
{
    this->globalObj = globalObject;
    this->commandHandler = handler;

    this->dataFile = new DataFile(globalObj->getDataDirectory()
                                  + "/commands.axb",
                                  this);

    // Left side
    this->syncCommandsButton = new QPushButton(QIcon::fromTheme("document-save",
                                                                QIcon(":/images/button-save.png")),
                                               tr("&Sync Commands"),
                                               this);
    connect(syncCommandsButton, SIGNAL(clicked()),
            this, SLOT(syncCommands()));



    this->commandListWidget = new QListWidget(this);
    commandListWidget->setDragDropMode(QAbstractItemView::InternalMove); // Movable items
    connect(commandListWidget, SIGNAL(currentRowChanged(int)),
            this, SLOT(onCommandSelected(int)));


    this->addCommandButton = new QPushButton(QIcon::fromTheme("list-add",
                                                              QIcon(":/images/list-add.png")),
                                             tr("&Add Command"),
                                             this);
    connect(addCommandButton, SIGNAL(clicked()),
            this, SLOT(addCommand()));

    this->removeCommandButton = new QPushButton(QIcon::fromTheme("list-remove",
                                                                 QIcon(":/images/list-remove.png")),
                                                tr("&Remove Command"),
                                                this);
    connect(removeCommandButton, SIGNAL(clicked()),
            this, SLOT(removeCommand()));



    // Right side
    this->triggerLineEdit = new QLineEdit(this);
    connect(triggerLineEdit, SIGNAL(textChanged(QString)),
            this, SLOT(onTriggerChanged(QString)));

    this->helpLineEdit = new QLineEdit(this);

    this->accessLevelSpinbox = new QSpinBox(this);
    accessLevelSpinbox->setRange(0, 100);
    accessLevelSpinbox->setSpecialValueText(tr("Anyone can use it",
                                               "'it' refers to a command"));

    this->typeCombobox = new QComboBox(this);
    typeCombobox->addItem(tr("Static Message"),
                          tr("Just enter a single sentence."));
    typeCombobox->addItem(tr("Random Sentence"),
                          tr("Add as many sentences as you want.\n"
                             "One of them will be selected at random as "
                             "a reply."));
    typeCombobox->addItem(tr("Random Line from Text File") + " (UTF-8)",
                          tr("A random line from the chosen file will be used "
                             "as a reply.\n"
                             "File encoding should be UTF-8."));
    typeCombobox->addItem(tr("Keyword-based Message"),
                          tr("Add keywords to the list, then enter the "
                             "corresponding definition for each keyword on "
                             "the other side."));
    typeCombobox->addItem(tr("Run Program"),
                          tr("Enter a command to be run, with any needed "
                             "parameters.\n"
                             "The reply will be the console output of the "
                             "program, unless you specify an output pattern, "
                             "and don't include the %output% variable "
                             "somewhere in it.",
                             "Don't translate %output%!"));
    typeCombobox->addItem(tr("Alias for another command"),
                          tr("Enter the original command that should be run "
                             "when this alias is used."));

    QFont helpFont;
    helpFont.setItalic(true);

    this->typeHelpLabel = new QLabel(this);
    typeHelpLabel->setFont(helpFont);
    typeHelpLabel->setWordWrap(true);
    typeHelpLabel->setContentsMargins(16, 1, 16, 1);
    typeHelpLabel->setSizePolicy(QSizePolicy::MinimumExpanding,
                                 QSizePolicy::Minimum);

    connect(typeCombobox, SIGNAL(currentIndexChanged(int)),
            this, SLOT(onTypeChanged(int)));


    this->staticLineEdit = new QLineEdit(this);

    this->randomListWidget = new StringListWidget(false, this);

    this->filenameLabel = new QLabel(this);
    this->filenameButton = new QPushButton(QIcon::fromTheme("document-open",
                                                            QIcon(":/images/button-edit.png")),
                                           tr("Select File..."),
                                           this);
    connect(filenameButton, SIGNAL(clicked()),
            this, SLOT(findFile()));
    this->filenameLayout = new QHBoxLayout();
    filenameLayout->addWidget(filenameLabel,  1);
    filenameLayout->addWidget(filenameButton, 0);


    this->keywordListWidget = new StringListWidget(true, this);
    keywordListWidget->setLowercaseItems(true);


    this->programLineEdit = new QLineEdit(this);

    this->aliasLineEdit = new QLineEdit(this);


    this->outputLineEdit = new QLineEdit(this);
    this->outputLineEdit->setPlaceholderText(tr("Optional, like: "
                                                "[RESULT] %line% %output%"));

    this->privateReplyCheckbox = new QCheckBox(this);



    this->saveCommandButton = new QPushButton(QIcon::fromTheme("document-save",
                                                               QIcon(":/images/button-save.png")),
                                              tr("Save &Command"),
                                              this);
    connect(saveCommandButton, SIGNAL(clicked()),
            this, SLOT(saveCommand()));


    // Layout
    this->leftLayout = new QVBoxLayout();
    leftLayout->addWidget(syncCommandsButton);
    leftLayout->addWidget(commandListWidget);
    leftLayout->addWidget(addCommandButton);
    leftLayout->addWidget(removeCommandButton);

    // These go in a sublayout to avoid spacing issues between them
    this->typeLayout = new QVBoxLayout();
    typeLayout->setContentsMargins(0, 0, 0, 0);
    typeLayout->setSpacing(0);
    typeLayout->addWidget(staticLineEdit);
    typeLayout->addWidget(randomListWidget);
    typeLayout->addLayout(filenameLayout);
    typeLayout->addWidget(keywordListWidget);
    typeLayout->addWidget(programLineEdit);
    typeLayout->addWidget(aliasLineEdit);


    this->formLayout = new QFormLayout();
    formLayout->addRow(tr("Command"),         triggerLineEdit);
    formLayout->addRow(tr("Help Text"),       helpLineEdit);
    formLayout->addRow(tr("Access Level"),    accessLevelSpinbox);
    formLayout->addRow(tr("Type"),            typeCombobox);
    formLayout->addWidget(typeHelpLabel);
    formLayout->addRow("", typeLayout);
    formLayout->addRow(tr("Output Pattern"),  outputLineEdit);
    formLayout->addRow(tr("Private Reply"),   privateReplyCheckbox);


    this->rightLayout = new QVBoxLayout();
    rightLayout->addLayout(formLayout);
    rightLayout->addStretch();
    rightLayout->addWidget(saveCommandButton, 0, Qt::AlignRight);

    this->rightWidget = new QWidget(this);
    rightWidget->setLayout(rightLayout);


    this->mainLayout = new QHBoxLayout();
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->addLayout(leftLayout,  1);
    mainLayout->addWidget(rightWidget, 5);
    this->setLayout(mainLayout);


    this->onTypeChanged(0); // Select first type to show its extra widget
    this->removeCommandButton->setDisabled(true);
    this->rightWidget->setDisabled(true);

    this->loadCommands();


    qDebug() << "CommandModule created";
}


CommandModule::~CommandModule()
{
    qDebug() << "CommandModule destroyed";
}




/*
 * Load commands from disk
 *
 */
void CommandModule::loadCommands()
{
    QVariantList commandDataList = this->dataFile->loadData();

    foreach (QVariant listItem, commandDataList)
    {
        QVariantMap map = listItem.toMap();
        QListWidgetItem *item = new QListWidgetItem();

        int commandType = map.value("type").toInt();

        item->setText(map.value("trigger").toString());
        item->setData(Qt::UserRole,     map.value("trigger"));
        item->setData(Qt::UserRole + 1, map.value("helpText"));
        item->setData(Qt::UserRole + 2, map.value("accessLevel"));
        item->setData(Qt::UserRole + 3, commandType);

        if (commandType == 0)
        {
            item->setData(Qt::UserRole + 4,
                          map.value("staticReply"));
        }
        else if (commandType == 1)
        {
            item->setData(Qt::UserRole + 4,
                          map.value("randomSentence"));
        }
        else if (commandType == 2)
        {
            item->setData(Qt::UserRole + 4,
                          map.value("filename"));
        }
        else if (commandType == 3)
        {
            item->setData(Qt::UserRole + 4,
                          map.value("keywords"));
        }
        else if (commandType == 4)
        {
            item->setData(Qt::UserRole + 4,
                          map.value("programToRun"));
        }
        else if (commandType == 5)
        {
            item->setData(Qt::UserRole + 4,
                          map.value("aliasFor"));
        }

        item->setData(Qt::UserRole + 5, map.value("outputPattern"));
        item->setData(Qt::UserRole + 6, map.value("privateReply"));

        this->commandListWidget->addItem(item);
    }

    this->commandHandler->setCommandData(commandDataList);

    this->globalObj->addToLog(tr("Loaded %Ln command(s).",
                                 "",
                                 commandDataList.length()));
}



bool CommandModule::validateCommand(int commandType)
{
    QString errorMessage;

    if (commandType == 0)
    {
        if (this->staticLineEdit->text().trimmed().isEmpty())
        {
            errorMessage = tr("Static reply is empty.");
            this->staticLineEdit->setFocus();
        }
    }
    else if (commandType == 1)
    {
        if (this->randomListWidget->getStringsFromList().length() == 0)
        {
            errorMessage = tr("There are no sentences in the list.");
            this->randomListWidget->setFocus();
        }
    }
    else if (commandType == 2)
    {
        if (this->textFilename.isEmpty())
        {
            errorMessage = tr("There is no file selected.");
            this->filenameButton->setFocus();
        }
    }
    else if (commandType == 3)
    {
        if (this->keywordListWidget->getStringsFromList().length() == 0)
        {
            errorMessage = tr("There are no keywords in the list.");
            this->keywordListWidget->setFocus();
        }

        // TODO -- FIXME: Check that each keyword has content
    }
    else if (commandType == 4)
    {
        if (this->programLineEdit->text().trimmed().isEmpty())
        {
            errorMessage = tr("There is no program specified.");
            this->programLineEdit->setFocus();
        }
    }
    else if (commandType == 5)
    {
        if (this->aliasLineEdit->text().trimmed().isEmpty())
        {
            errorMessage = tr("You need to enter the command this alias "
                              "will execute.");
            this->aliasLineEdit->setFocus();
        }
        else if (this->aliasLineEdit->text().trimmed()
              == this->triggerLineEdit->text().trimmed())
        {
            errorMessage = tr("A command can't be an alias of itself.");
            this->aliasLineEdit->setFocus();
        }
    }


    if (!errorMessage.isEmpty())
    {
        QMessageBox::warning(this,
                             tr("Error validating fields"),
                             errorMessage);
        return false;
    }

    return true;
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////




void CommandModule::onTriggerChanged(QString newTrigger)
{
    this->saveCommandButton->setDisabled(newTrigger.trimmed().isEmpty());
}


void CommandModule::findFile()
{
    QString filename = QFileDialog::getOpenFileName(this,
                                                    tr("Select a text file"),
                                                    QDir::homePath(),
                                                    "*");
    if (!QFile(filename).exists())
    {
        filename.clear();
        this->filenameLabel->setText(tr("Selected file does not exist."));
    }

    if (!filename.isEmpty())
    {
        this->textFilename = filename;
        this->filenameLabel->setText(filename);
    }
}



void CommandModule::addCommand()
{
    QListWidgetItem *item = new QListWidgetItem("-- "
                                                + tr("New Command")
                                                + " --");
    this->commandListWidget->addItem(item);

    int lastRow = commandListWidget->count() - 1;
    commandListWidget->setCurrentRow(lastRow);
    commandListWidget->setCurrentItem(item);
    item->setSelected(true);

    this->saveCommandButton->setDisabled(true); // Until something's entered
    this->triggerLineEdit->setFocus();
}


void CommandModule::removeCommand()
{
    int row = this->commandListWidget->currentRow();
    if (row != -1)
    {
        QListWidgetItem *item = this->commandListWidget->takeItem(row);
        delete item;
    }

    if (commandListWidget->count() == 0) // No commands left
    {
        this->removeCommandButton->setDisabled(true);
        this->rightWidget->setDisabled(true);
    }
}


void CommandModule::onCommandSelected(int row)
{
    if (row < 0) // Small protection
    {
        return;
    }

    QListWidgetItem *item = commandListWidget->item(row);

    this->triggerLineEdit->setText(item->data(Qt::UserRole).toString());
    this->helpLineEdit->setText(item->data(Qt::UserRole + 1).toString());
    this->accessLevelSpinbox->setValue(item->data(Qt::UserRole + 2).toInt());
    this->typeCombobox->setCurrentIndex(item->data(Qt::UserRole + 3).toInt());

    this->staticLineEdit->clear();
    this->randomListWidget->clearList();
    this->filenameLabel->setText("[" + tr("No file selected") + "]");
    this->textFilename.clear();
    this->keywordListWidget->clearList();
    this->programLineEdit->clear();
    this->aliasLineEdit->clear();


    int commandType = typeCombobox->currentIndex();
    if (commandType == 0)
    {
        this->staticLineEdit->setText(item->data(Qt::UserRole + 4).toString());
    }
    else if (commandType == 1)
    {
        QStringList stringList = item->data(Qt::UserRole + 4).toStringList();
        this->randomListWidget->addStringsToList(stringList);
    }
    else if (commandType == 2)
    {
        this->textFilename = item->data(Qt::UserRole + 4).toString();
        this->filenameLabel->setText(this->textFilename);
    }
    else if (commandType == 3)
    {
        QVariantMap stringMap = item->data(Qt::UserRole + 4).toMap();
        this->keywordListWidget->addItemsToList(stringMap);
    }
    else if (commandType == 4)
    {
        this->programLineEdit->setText(item->data(Qt::UserRole + 4).toString());
    }
    else if (commandType == 5)
    {
        this->aliasLineEdit->setText(item->data(Qt::UserRole + 4).toString());
    }


    this->outputLineEdit->setText(item->data(Qt::UserRole + 5).toString());
    this->privateReplyCheckbox->setChecked(item->data(Qt::UserRole + 6).toBool());


    this->removeCommandButton->setEnabled(true);
    this->rightWidget->setEnabled(true);
}


void CommandModule::onTypeChanged(int commandType)
{
    // Show the informational text for this command type
    this->typeHelpLabel->setText(typeCombobox->itemData(commandType).toString());

    this->staticLineEdit->hide();
    this->randomListWidget->hide();
    this->filenameLabel->hide();
    this->filenameButton->hide();
    this->keywordListWidget->hide();
    this->programLineEdit->hide();
    this->aliasLineEdit->hide();


    if (commandType == 0)
    {
        this->staticLineEdit->show();
    }
    else if (commandType == 1)
    {
        this->randomListWidget->show();
    }
    else if (commandType == 2)
    {
        this->filenameLabel->show();
        this->filenameButton->show();
    }
    else if (commandType == 3)
    {
        this->keywordListWidget->show();
    }
    else if (commandType == 4)
    {
        this->programLineEdit->show();
    }
    else if (commandType == 5)
    {
        this->aliasLineEdit->show();
    }
}


void CommandModule::saveCommand()
{
    QString newTrigger = triggerLineEdit->text().trimmed();
    int commandType = typeCombobox->currentIndex();

    // Check for duplicates
    for (int counter = 0; counter < commandListWidget->count(); ++counter)
    {
        QListWidgetItem *commandItem = commandListWidget->item(counter);
        if (commandItem->data(Qt::UserRole).toString() == newTrigger)
        {
            if (commandItem != this->commandListWidget->currentItem())
            {
                QMessageBox::warning(this,
                                     tr("Command exists"),
                                     tr("Error: There is already a command "
                                        "in the list with this trigger."));
                this->triggerLineEdit->setFocus();

                return;
            }
        }
    }


    // Check minimum required fields are filled
    if (!validateCommand(commandType))
    {
        return;
    }

    QListWidgetItem *item = this->commandListWidget->currentItem();

    triggerLineEdit->setText(newTrigger); // Show it as it will be stored
    item->setText(newTrigger);
    item->setData(Qt::UserRole,     newTrigger);
    item->setData(Qt::UserRole + 1, helpLineEdit->text().trimmed());
    item->setData(Qt::UserRole + 2, accessLevelSpinbox->value());
    item->setData(Qt::UserRole + 3, commandType);


    if (commandType == 0)
    {
        item->setData(Qt::UserRole + 4,
                      this->staticLineEdit->text().trimmed());
    }
    else if (commandType == 1)
    {
        item->setData(Qt::UserRole + 4,
                      this->randomListWidget->getStringsFromList());
    }
    else if (commandType == 2)
    {
        item->setData(Qt::UserRole + 4,
                      this->textFilename);
    }
    else if (commandType == 3)
    {
        item->setData(Qt::UserRole + 4,
                      this->keywordListWidget->getItemsFromList());
    }
    else if (commandType == 4)
    {
        item->setData(Qt::UserRole + 4,
                      this->programLineEdit->text().trimmed());
    }
    else if (commandType == 5)
    {
        item->setData(Qt::UserRole + 4,
                      this->aliasLineEdit->text().trimmed());
    }

    item->setData(Qt::UserRole + 5, this->outputLineEdit->text().trimmed());
    item->setData(Qt::UserRole + 6, this->privateReplyCheckbox->isChecked());


    this->commandListWidget->setFocus();
}


/*
 * Synchronize commands with the CommandHandler,
 * and also save them to disk
 *
 */
void CommandModule::syncCommands()
{
    QVariantList commandDataList;

    for (int counter = 0; counter < commandListWidget->count(); ++counter)
    {
        QListWidgetItem *item = commandListWidget->item(counter);
        int commandType = item->data(Qt::UserRole + 3).toInt();

        QVariantMap commandMap;
        commandMap.insert("trigger",     item->data(Qt::UserRole));
        commandMap.insert("helpText",    item->data(Qt::UserRole + 1));
        commandMap.insert("accessLevel", item->data(Qt::UserRole + 2));
        commandMap.insert("type",        commandType);

        if (commandType == 0) // Static reply
        {
            commandMap.insert("staticReply", item->data(Qt::UserRole + 4));
        }
        else if (commandType == 1) // List of strings, to choose one at random
        {
            commandMap.insert("randomSentence", item->data(Qt::UserRole + 4));
        }
        else if (commandType == 2) // A text file, to choose a line at random
        {
            commandMap.insert("filename", item->data(Qt::UserRole + 4));
        }
        else if (commandType == 3) // A list of key:value pairs
        {
            commandMap.insert("keywords", item->data(Qt::UserRole + 4));
        }
        else if (commandType == 4) // A program to run
        {
            commandMap.insert("programToRun", item->data(Qt::UserRole + 4));
        }
        else if (commandType == 5) // Act as an alias for another command
        {
            commandMap.insert("aliasFor", item->data(Qt::UserRole + 4));
        }

        commandMap.insert("outputPattern", item->data(Qt::UserRole + 5));
        commandMap.insert("privateReply",  item->data(Qt::UserRole + 6));

        commandDataList.append(commandMap);
    }


    this->commandHandler->setCommandData(commandDataList);
    this->dataFile->saveData(commandDataList);

    this->globalObj->addToLog(tr("Saved %Ln command(s).", "",
                              commandDataList.length()));
}

