/*
 *   This file is part of AkariXB
 *   Copyright 2015-2016  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "generalmodule.h"


GeneralModule::GeneralModule(GlobalObject *globalObject,
                             QWidget *parent) : QWidget(parent)
{
    this->globalObj = globalObject;

    this->xmppClient = this->globalObj->getXmppClient();
    connect(xmppClient, SIGNAL(connected()),
            this, SLOT(onConnected()));
    connect(xmppClient, SIGNAL(disconnected()),
            this, SLOT(onDisconnected()));


    QFont infoFont;
    infoFont.setBold(true);
    infoFont.setPointSize(infoFont.pointSize() + 4);

    this->infoLabel = new QLabel(tr("Offline"), this);
    infoLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
    infoLabel->setFont(infoFont);
    infoLabel->setAlignment(Qt::AlignCenter);


    infoFont.setPointSize(infoFont.pointSize() - 2);

    this->activityLabel = new QLabel("--", this);
    activityLabel->setFont(infoFont);
    activityLabel->setWordWrap(true);
    activityLabel->setContentsMargins(4, 4, 4, 4);
    activityLabel->setSizePolicy(QSizePolicy::MinimumExpanding,
                                 QSizePolicy::Minimum);
    connect(globalObj, SIGNAL(activityChanged(QString)),
            activityLabel, SLOT(setText(QString)));

    this->statusMessageLabel = new QLabel("--", this);
    statusMessageLabel->setFont(infoFont);
    statusMessageLabel->setWordWrap(true);
    statusMessageLabel->setContentsMargins(4, 4, 4, 4);
    statusMessageLabel->setSizePolicy(QSizePolicy::MinimumExpanding,
                                      QSizePolicy::Minimum);
    connect(globalObj, SIGNAL(statusMessageChanged(QString)),
            statusMessageLabel, SLOT(setText(QString)));




    this->tmpPresenceBrowser = new QTextBrowser(this);


    this->connectButton = new QPushButton(QIcon::fromTheme("network-connect",
                                                           QIcon(":/images/button-online.png")),
                                          tr("Connect"),
                                          this);
    connect(connectButton, SIGNAL(clicked()),
            this, SIGNAL(connectionRequested()));


    this->disconnectButton = new QPushButton(QIcon::fromTheme("network-disconnect",
                                                              QIcon(":/images/button-offline.png")),
                                             tr("Disconnect"),
                                             this);
    this->disconnectButton->hide();
    connect(disconnectButton, SIGNAL(clicked()),
            xmppClient, SLOT(disconnectFromServer()));


    // Layout
    this->formLayout = new QFormLayout();
    formLayout->addRow(tr("Current Activity:"), activityLabel);
    formLayout->addRow(tr("Status Message:"),   statusMessageLabel);


    this->mainLayout = new QVBoxLayout();
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->addStretch(1);
    mainLayout->addWidget(infoLabel,          1);
    mainLayout->addStretch(1);
    mainLayout->addLayout(formLayout,         1);
    mainLayout->addStretch(1);
    mainLayout->addWidget(tmpPresenceBrowser, 0);
    mainLayout->addWidget(connectButton,      0, Qt::AlignRight);
    mainLayout->addWidget(disconnectButton,   0, Qt::AlignRight);
    this->setLayout(mainLayout);

    qDebug() << "GeneralModule created";
}

GeneralModule::~GeneralModule()
{
    qDebug() << "GeneralModule destroyed";
}


void GeneralModule::setInfoMessage(QString message)
{
    this->infoLabel->setText(message);
}


void GeneralModule::setConnecting()
{
    this->connectButton->setDisabled(true);
}


void GeneralModule::addPresence(QXmppPresence presence)
{
    // TMP FIXME
    QString availableType;
    if (presence.type() == QXmppPresence::Available)
    {
        switch (presence.availableStatusType())
        {
        case QXmppPresence::Away:
            availableType = "Away";
            break;
        case QXmppPresence::XA:
            availableType = "Extended away";
            break;
        case QXmppPresence::DND:
            availableType = "Do not disturb";
            break;
        case QXmppPresence::Chat:
            availableType = "Free for chat";
            break;

        default:
            availableType = "Online";
        }
    }
    else
    {
        availableType = "Offline";
    }

    if (!presence.statusText().isEmpty())
    {
        availableType.append(" (" + presence.statusText() + ")"); // FIXME, statusText could be "%3" or something
    }

    QString presenceString = QString("%1 is %2 [Priority: %3, Type: %4]")
                             .arg(presence.from())
                             .arg(availableType)
                             .arg(presence.priority())
                             .arg(presence.type());


    if (presence.type() == QXmppPresence::Subscribe)
    {
        presenceString = presence.from() + " wants to subscribe to your presence.";
    }
    else if (presence.type() == QXmppPresence::Subscribed)
    {
        presenceString = presence.from() + " accepted your subscription.";
    }
    else if (presence.type() == QXmppPresence::Unsubscribe)
    {
        presenceString = presence.from() + " Unsubscribed from your presence.";
    }
    else if (presence.type() == QXmppPresence::Unsubscribed)
    {
        presenceString = presence.from() + " removed your authorization to their presence.";
    }


    this->tmpPresenceBrowser->append(presenceString);
}


//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


void GeneralModule::onConnected()
{
    this->connectButton->hide();
    this->connectButton->setEnabled(true);
    this->disconnectButton->show();
}


void GeneralModule::onDisconnected()
{
    this->connectButton->show();
    this->disconnectButton->hide();
}

